/*--------------------------------------------------------------------
 * FICHERO:  CalcuHex.c
 * OBJETIVO: Mdulo principal de "CalcuHex"
 * AUTOR:    DJ Delorie
 * FECHA:    J.2.7.1998
 *------------------------------------------------------------------*/

/* Copyright 1998 DJ Delorie <dj@delorie.com>
   Distributed under the terms of the GNU GPL
   http://www.delorie.com/store/hcalc/
*/

/* Adaptacin a MS Windows 3.1 y traduccin al
   espaol de Pedro Reina <pedro@anit.es>
   http://www.anit.es/pedro
*/

/*--------------------------------------------------------------------
 * Ficheros de cabecera
 *------------------------------------------------------------------*/

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include "CalcuHex.h"

/*--------------------------------------------------------------------
 * Definicin de tipos
 *------------------------------------------------------------------*/

typedef unsigned long entero;

/*--------------------------------------------------------------------
 * Definicin de macros constantes
 *------------------------------------------------------------------*/

#define ANCHO 125
#define ALTO  147

#define CARACTERES_IZQUIERDA  6
#define CARACTERES_ARRIBA     6

#define BITS_DERECHA 92
#define BITS_ARRIBA   6

#define MAXENTRADA  40
#define TAMPANTALLA 32

/*--------------------------------------------------------------------
 * Definicin de macros funciones
 *------------------------------------------------------------------*/

/*--------------------------------------------------------------------
 * MACRO:    AnchoPantalla()
 * OBJETIVO: Calcular el ancho total de la pantalla
 *------------------------------------------------------------------*/
#define AnchoPantalla()  GetSystemMetrics(SM_CXSCREEN)

/*--------------------------------------------------------------------
 * MACRO:    AltoPantalla()
 * OBJETIVO: Calcular el alto total de la pantalla
 *------------------------------------------------------------------*/
#define AltoPantalla()  GetSystemMetrics(SM_CYSCREEN)

/*--------------------------------------------------------------------
 * Declaracin de funciones
 *------------------------------------------------------------------*/

BOOL IniciaAplicacion (HANDLE,int);
LONG FAR PASCAL ProcedimientoPrincipal (HWND,UINT,WPARAM,LPARAM);
BOOL CentraVentana (HWND);
int  DibujaVentana (void);
void DibujaBits (HDC,HDC);
void DibujaCaracteres (HDC,HDC);
void Raton (LPARAM);
void Tecla (WPARAM);
void EscribeBits (void);
void EscribeCadena (char *);
void CalculaValor (void);
void MuestraValor (void);
void ComienzaNumero (void);
void TerminaNumero (void);
void Copia (void);

/*--------------------------------------------------------------------
 * Definicin de variables globales
 *------------------------------------------------------------------*/

char    NombreAplicacion[10];
HBITMAP Calculadora, Caracteres, Bits;

HWND VentanaAplicacion;

char MapaCaracteres[] = " 0123456789ABCDEF-x,.ro+";
int  PosicionCaracter[256];
char Desplazamiento[15];

entero Mascara;
BOOL   MuestraBits;

char OperacionPendiente = 0;
int  HaciendoNumero = 0;
int  Base = 10;

char   Pantalla[TAMPANTALLA+1];
char   Entrada[MAXENTRADA];
int    PosicionEntrada;
double Valor, ValorPendiente, ValorMemoria=0;

char *Traduccion[] = {
  "PPPP\033",
  "DHOB\010",
  "trm<>",
  "Sdef/",
  "~abc*",
  "|789-",
  "&456+",
  "^123=",
  "u0._="
  };

/*--------------------------------------------------------------------
 * Programa principal
 *------------------------------------------------------------------*/
#pragma argsused
int PASCAL
WinMain
  (
  HANDLE Instancia,    // Manejador de instancia de aplicacin
  HANDLE Anterior,     // Manejador de instancia anterior
  LPSTR  Parametros,   // Cadena con la lnea de parmetros
  int    ModoMostrar   // Aplicacin como ventana o como icono
  )
  {
  MSG Mensaje;

  if ( !IniciaAplicacion (Instancia, ModoMostrar) )
    { return FALSE; }

  while ( GetMessage (&Mensaje, NULL, 0, 0) )
    {
    TranslateMessage (&Mensaje);
    DispatchMessage (&Mensaje);
    }

  return ( Mensaje.wParam );
  }

/*--------------------------------------------------------------------
 * Definicin de funciones
 *------------------------------------------------------------------*/

/*--------------------------------------------------------------------
 * FUNCION:  IniciaAplicacion()
 * OBJETIVO: Arrancar la aplicacin, registrar la clase de ventana
 *           y crear la ventana principal
 *------------------------------------------------------------------*/
static BOOL
IniciaAplicacion
  (
  HINSTANCE Instancia,
  int       ModoMostrar
  )
  {
  int      i;
  WNDCLASS Clase;
  HWND     VentanaPrincipal;
  HMENU    MenuControl;

  strcpy (NombreAplicacion,"CalcuHex");

  // Registramos la clase de ventana
  Clase.style	      = CS_HREDRAW | CS_VREDRAW;
  Clase.lpfnWndProc   = ProcedimientoPrincipal;
  Clase.cbClsExtra    = 0;
  Clase.cbWndExtra    = 0;
  Clase.hInstance     = Instancia;
  Clase.hIcon	      = LoadIcon (Instancia, NombreAplicacion);
  Clase.hCursor	      = LoadCursor (NULL, IDC_ARROW);
  Clase.hbrBackground = (HBRUSH)(COLOR_WINDOW + 1);
  Clase.lpszMenuName  = NULL;
  Clase.lpszClassName = NombreAplicacion;

  if ( !RegisterClass (&Clase) )
    { return ( FALSE ); }

  // Creamos la ventana principal
  VentanaPrincipal = CreateWindowEx (WS_EX_TOPMOST,
    NombreAplicacion, NombreAplicacion,
    WS_OVERLAPPED|WS_CAPTION|WS_SYSMENU|WS_MINIMIZEBOX,
    0, 0,
    ANCHO+GetSystemMetrics(SM_CXBORDER),
    ALTO+GetSystemMetrics(SM_CYCAPTION),
    NULL, NULL, Instancia, NULL);

  if ( !VentanaPrincipal )
    { return ( FALSE ); }

  // Cargamos los bitmaps
  Calculadora  = LoadBitmap (Instancia, MAKEINTRESOURCE(IDB_CALCUHEX));
  Caracteres = LoadBitmap (Instancia, MAKEINTRESOURCE(IDB_CARACTER));
  Bits  = LoadBitmap (Instancia, MAKEINTRESOURCE(IDB_BITS));

  // Damos valores iniciales
  for ( i=0 ; i<256 ; i++ )
    { PosicionCaracter[i] = 0; }
  for ( i=0 ; MapaCaracteres[i] ; i++ )
    { PosicionCaracter[MapaCaracteres[i]] = i*6; }

  // Cambiamos el men del cuadro de control
  MenuControl = GetSystemMenu (VentanaPrincipal, FALSE);
  DeleteMenu (MenuControl, SC_SIZE,     MF_BYCOMMAND);
  DeleteMenu (MenuControl, SC_MAXIMIZE, MF_BYCOMMAND);

  // Mostramos la ventana
  CentraVentana (VentanaPrincipal);
  ShowWindow (VentanaPrincipal, ModoMostrar);

  return ( TRUE );
  }

/*--------------------------------------------------------------------
 * FUNCION:  CentraVentana()
 * OBJETIVO: Centrar una ventana en pantalla
 * ENTRADAS: La ventana
 * SALIDAS:  Lgica, que indica si ha cambiado la posicin de la ventana
 * NOTA:     Slo se cambia la posicin
 *------------------------------------------------------------------*/
BOOL
CentraVentana
  (
  HWND Ventana
  )
  {
  BOOL Respuesta = FALSE;
  RECT Info;
  int  AnchoVentana, AltoVentana, Izquierda, Arriba;

  GetWindowRect (Ventana, &Info);

  AnchoVentana = Info.right - Info.left;
  AltoVentana = Info.bottom - Info.top;

  Izquierda = (AnchoPantalla()-AnchoVentana) / 2;
  Arriba = (AltoPantalla()-AltoVentana) / 2;

  if ( Izquierda >= 0 && Arriba >= 0 )
    {
    SetWindowPos (Ventana, HWND_TOP, Izquierda, Arriba, 0, 0,
                  SWP_NOACTIVATE | SWP_NOSIZE | SWP_NOREDRAW | SWP_NOZORDER);
    Respuesta = TRUE;
    }

  return ( Respuesta );
  }

/*--------------------------------------------------------------------
 * FUNCION:  ProcedimientoPrincipal()
 * OBJETIVO: Recibir los mensajes de Windows para la clase
 *           de ventanas principal de esta aplicacin
 *------------------------------------------------------------------*/
LONG FAR PASCAL
ProcedimientoPrincipal
  (
  HWND   Ventana,        // Ventana a la que se enva el mensaje
  UINT   Mensaje,        // El mensaje
  WPARAM ParametroCorto, // Informacin adicional
  LPARAM ParametroLargo  // Informacin adicional
  )
  {
  VentanaAplicacion = Ventana;

  switch ( Mensaje )
    {
    case WM_DESTROY: PostQuitMessage(0); break;

    case WM_PAINT: return DibujaVentana();

    case WM_LBUTTONDOWN:
    case WM_RBUTTONDOWN:
      Raton (ParametroLargo);
      break;

    case WM_CHAR: Tecla (ParametroCorto); break;
    }

  return ( DefWindowProc (Ventana, Mensaje,
                          ParametroCorto, ParametroLargo) );
  }

/*--------------------------------------------------------------------
 * FUNCION:  DibujaVentana()
 * OBJETIVO: Dibujar el contenido de la ventana
 *------------------------------------------------------------------*/
int
DibujaVentana (void)
  {
  PAINTSTRUCT Pintura;
  HDC Contexto = BeginPaint (VentanaAplicacion, &Pintura);
  HDC Auxiliar = CreateCompatibleDC (Contexto);

  // Dibujamos la calculadora
  SelectObject (Auxiliar, Calculadora);
  BitBlt (Contexto, 0, 0, ANCHO, ALTO, Auxiliar, 0, 0, SRCCOPY);

  // Dibujamos la pantalla de la calculadora
  if ( MuestraBits )  { DibujaBits (Contexto, Auxiliar); }
  else                { DibujaCaracteres (Contexto, Auxiliar); }

  DeleteDC (Auxiliar);
  EndPaint (VentanaAplicacion, &Pintura);

  return ( 0 );
  }

/*--------------------------------------------------------------------
 * FUNCION:  DibujaBits()
 * OBJETIVO: Dibujar la pantalla de la calculadora con bits
 *------------------------------------------------------------------*/
void
DibujaBits
  (
  HDC Contexto,
  HDC Auxiliar
  )
  {
  int    i;
  entero Bit=0x80000000;

  SelectObject (Auxiliar, Bits);
  for ( i=0 ; i<32 ; i++ )
    {
    if ( Mascara & Bit )  { Pantalla[i] = '1'; }
    else                  { Pantalla[i] = '0'; }
    Bit >>= 1;

    BitBlt (Contexto, BITS_DERECHA-2*i-3*(i/4), BITS_ARRIBA, 1, 7,
	    Auxiliar, (Mascara >> i) & 1, 0, SRCCOPY);
    }
  Pantalla[i] = NULL;
  }

/*--------------------------------------------------------------------
 * FUNCION:  DibujaCaracteres()
 * OBJETIVO: Dibujar la pantalla de la calculadora con caracteres
 *------------------------------------------------------------------*/
void
DibujaCaracteres
  (
  HDC Contexto,
  HDC Auxiliar
  )
  {
  int i;

  SelectObject (Auxiliar, Caracteres);
  for ( i=0 ; i<15 ; i++)
    {
    BitBlt (Contexto, CARACTERES_IZQUIERDA+6*i, CARACTERES_ARRIBA, 5, 7,
	    Auxiliar, Desplazamiento[i], 0, SRCCOPY);
    }
  }

/*--------------------------------------------------------------------
 * FUNCION:  Raton()
 * OBJETIVO: Procesar un mensaje de botn de ratn pulsado
 *------------------------------------------------------------------*/
void
Raton
  (
  LPARAM Parametro
  )
  {
  int x, y;

  x = LOWORD (Parametro);
  y = HIWORD (Parametro);

  x = (x-2)/24;
  if (x < 0)  { x = 0; }
  if (x > 4)  { x = 4; }

  y = (y-1)/16;
  if (y < 0)  { y = 0; }
  if (y > 8)  { y = 8; }

  Tecla (Traduccion[y][x]);
  }

/*--------------------------------------------------------------------
 * FUNCION:  Tecla()
 * OBJETIVO: Procesar un mensaje de tecla pulsada
 *------------------------------------------------------------------*/
void
Tecla
  (
  WPARAM Tecla
  )
  {
  int v = Tecla;

  switch (Tecla)
    {
    case 27: /* CLR, ESC */
      HaciendoNumero = 0;
      PosicionEntrada = 0;
      OperacionPendiente = 0;
      Valor = ValorPendiente = 0;
      EscribeCadena("");
      break;

    case 8: /* DEL */
      if ( HaciendoNumero )
        {
        if ( PosicionEntrada==1 )
          { MessageBeep (-1); }
        else
          {
          Entrada[--PosicionEntrada] = 0;
          CalculaValor();
          MuestraValor();
          }
        }
      break;

    case 'u':  /* CE */
    case 'U':
      if ( HaciendoNumero )
        {
        HaciendoNumero = 0;
        EscribeCadena("");
        }
      break;

    case 'a':
    case 'b':
    case 'c':
    case 'd':
    case 'e':
    case 'f':
      v = Tecla - 'a' - '9' - 1;
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
      v -= '0';
      if ( v >= Base || PosicionEntrada == MAXENTRADA-1 )
        { MessageBeep (-1); }
      else
        {
        ComienzaNumero();
        Entrada[PosicionEntrada++] = Tecla;
        Entrada[PosicionEntrada] = 0;
        CalculaValor();
        MuestraValor();
        }
      break;

    case '.':
      if ( strchr(Entrada, '.') )
        {
        MessageBeep (-1);
        break;
        }

    case ',':
      if (PosicionEntrada == 1 || PosicionEntrada == MAXENTRADA-1)
        { MessageBeep (-1); }
      else
        {
        ComienzaNumero();
        Entrada[PosicionEntrada++] = Tecla;
        Entrada[PosicionEntrada] = 0;
        CalculaValor();
        MuestraValor();
        }
      break;

    case '_':   /* +/- */
      if ( HaciendoNumero )
        {
        if ( Entrada[0]=='-' )  { Entrada[0] = ' '; }
        else                    { Entrada[0] = '-'; }
        CalculaValor();
        MuestraValor();
        }
      else
        {
        Valor *= -1.0;
        ValorPendiente *= -1.0;
        MuestraValor();
        }
      break;

    case 'D':
      TerminaNumero();
      Base = 10;
      MuestraValor();
      break;

    case 'H':
      TerminaNumero();
      Base = 16;
      MuestraValor();
      break;

    case 'O':
      TerminaNumero();
      Base = 8;
      MuestraValor();
      break;

    case 'B':
      TerminaNumero();
      Base = 2;
      MuestraValor();
      break;

    case 'x':
      Tecla = '*';
    case '+':
    case '-':
    case '*':
    case '/':
    case '^':
    case '&':
    case '|':
    case 'S':
    case '=':
      TerminaNumero();
      OperacionPendiente = Tecla;
      break;

    case 13:
    case 10:
      TerminaNumero();
      break;

    case '~':
      TerminaNumero();
      Valor = (double)(~(entero)Valor);
      MuestraValor();
      break;

    case '<':
      TerminaNumero();
      Valor = (double)((entero)Valor << 1);
      MuestraValor();
      break;

    case '>':
      TerminaNumero();
      Valor = (double)((entero)Valor >> 1);
      MuestraValor();
      break;

    case 't': /* STO */
    case 'T':
      ValorMemoria = Valor;
      break;

    case 'r': /* RCL */
    case 'R':
      Valor = ValorMemoria;
      MuestraValor();
      HaciendoNumero = 1;
      PosicionEntrada = 1;
      Entrada[0] = ' ';
      break;

    case 'm': /* SUM */
    case 'M':
      ValorMemoria += Valor;
      break;

    case 'P': /* Han pulsado sobre la pantalla de la calculadora */
    case 'p':
      Copia();
      break;
    }
  }

/*--------------------------------------------------------------------
 * FUNCION:  EscribeBits()
 * OBJETIVO: Escribir en la pantalla de la calculadora un nmero
 *           en notacin de bits
 *------------------------------------------------------------------*/
void
EscribeBits (void)
  {
  MuestraBits = TRUE;
  InvalidateRect (VentanaAplicacion, NULL, FALSE);
  }

/*--------------------------------------------------------------------
 * FUNCION:  EscribeCadena()
 * OBJETIVO: Escribir en la pantalla de la calculadora una cadena
 *------------------------------------------------------------------*/
void
EscribeCadena
  (
  char *Cadena
  )
  {
  char Aux[16];
  int i;

  if ( Cadena[0]!=' ' )  { strcpy (Pantalla, Cadena); }
  else                   { strcpy (Pantalla, Cadena+1); }

  sprintf (Aux, "%15.15s", Cadena);
  for ( i=0 ; i<15 ; i++ )
    { Desplazamiento[i] = PosicionCaracter[Aux[i]]; }
  MuestraBits = FALSE;
  InvalidateRect (VentanaAplicacion, NULL, FALSE);
  }

/*--------------------------------------------------------------------
 * FUNCION:  CalculaValor()
 * OBJETIVO: Calcular el nmero Valor a partir de la cadena Entrada
 *------------------------------------------------------------------*/
void
CalculaValor (void)
  {
  char  *Posicion = Entrada;
  double Escala = 1;
  int    Signo = +1;

  if ( *Posicion == '-' )  { Signo = -1; }
  Valor = 0;
  while (*++Posicion)
    {
    if (*Posicion == '.')  { break; }
    if ( *Posicion >= '0' && *Posicion <= '9' )
      {
      Valor *= Base;
      Valor += *Posicion-'0';
      }
    if (*Posicion >= 'a' && *Posicion <= 'f')
      {
      Valor *= Base;
      Valor += *Posicion-'a'+10;
      }
    }
  if ( *Posicion )
    {
    while ( *++Posicion )
      {
      if ( *Posicion >= '0' && *Posicion <= '9' )
        {
	Escala *= Base;
	Valor += (*Posicion-'0')/Escala;
        }
      if (*Posicion >= 'a' && *Posicion <= 'f')
        {
	Escala *= Base;
	Valor += (*Posicion-'a'+10)/Escala;
        }
      }
    }
  Valor *= Signo;
  }

/*--------------------------------------------------------------------
 * FUNCION:  MuestraValor()
 * OBJETIVO: Escribir en la pantalla de la calculadora el Valor
 *           actual con la notacin apropiada
 *------------------------------------------------------------------*/
void
MuestraValor (void)
  {
  static char Hexadecimal[] = "0123456789ABCDEF";
  char   Aux[20], Comas[40], Tmp;
  char  *Posicion, *PosComas, *PosPunto, *Inverso;
  entero Largo;
  double v = Valor;

  if ( Base == 2 )
    {
    Mascara = (entero)v & 0xFFFFFFFFL;
    EscribeBits();
    return;
    }

  Aux[0] = ' ';
  if ( v<0 )
    {
    Aux[0] = '-';
    v = -v;
    }

  if ( Base==10 )
    {
    sprintf (Aux+1, "%.14G", v);
    if ( strchr(Aux+1, 'E') )  { sprintf (Aux+1, "%.9G", v); }
    if ( Aux[14] == '.' )  { Aux[14] = 0; }
    }
  else
    {
    Largo = (entero)v;
    Posicion = Aux+1;
    if ( Base==16 )
      {
      *Posicion++ = '0';
      *Posicion++ = 'x';
      }
    else if ( Base==8 )
      { *Posicion++ = '0'; }

    Inverso = Posicion;
    do {
       *Posicion++ = Hexadecimal[Largo%Base];
       Largo /= Base;
       } while ( Largo );
    *Posicion-- = 0;
    while ( Inverso<Posicion )
      {
      Tmp = *Inverso;
      *Inverso = *Posicion;
      *Posicion = Tmp;
      Posicion--;
      Inverso++;
      }
    }

  PosComas = Comas + 40;
  Posicion = Aux + strlen(Aux);
  PosPunto = strchr (Aux, '.');
  if ( PosPunto==0 )  { PosPunto = Posicion; }

  *--PosComas = 0;
  while ( Posicion>=Aux )
    {
    *--PosComas = *Posicion--;
    switch (Base)
      {
      case 10:
        if ( isdigit(PosComas[0]) &&
             isdigit(PosComas[1]) &&
             isdigit(PosComas[2]) &&
             Posicion<PosPunto    &&
             Posicion>=Aux        &&
             isdigit(*Posicion) )
          { *--PosComas = ','; }
        break;

      case 16:
        if ( isxdigit(PosComas[0]) &&
             isxdigit(PosComas[1]) &&
             isxdigit(PosComas[2]) &&
             isxdigit(PosComas[3]) &&
             Posicion>=Aux         &&
             isxdigit(*Posicion) )
          { *--PosComas = ','; }
        break;
      }
    }

  if ( strlen(PosComas)>15 )  { EscribeCadena (Aux); }
  else                        { EscribeCadena (PosComas); }
  }

/*--------------------------------------------------------------------
 * FUNCION:  TerminaNumero()
 * OBJETIVO: Dar por concluida la introduccin de un nmero y
 *           calcular la operacin pendiente
 *------------------------------------------------------------------*/
void
TerminaNumero (void)
  {
  if ( !HaciendoNumero )  { return; }

  HaciendoNumero = 0;
  PosicionEntrada = 0;

  switch ( OperacionPendiente )
    {
    case '+':
      Valor = ValorPendiente + Valor;
      break;

    case '-':
      Valor = ValorPendiente - Valor;
      break;

    case '*':
      Valor = ValorPendiente * Valor;
      break;

    case '/':
      Valor = ValorPendiente / Valor;
      break;

    case '&':
      Valor = (double)((entero)ValorPendiente & (entero)Valor);
      break;

    case '|':
      Valor = (double)((entero)ValorPendiente | (entero)Valor);
      break;

    case '^':
      Valor = (double)((entero)ValorPendiente ^ (entero)Valor);
      break;

    case 'S':
      if ( Valor<0 )
        { Valor = (double)((entero)ValorPendiente >> (entero)(-Valor)); }
      else
        { Valor = (double)((entero)ValorPendiente << (entero)Valor); }
      break;
    }

  ValorPendiente = Valor;
  OperacionPendiente = 0;
  MuestraValor();
  }

/*--------------------------------------------------------------------
 * FUNCION:  ComienzaNumero()
 * OBJETIVO: Prepararnos para comenzar a recibir un nmero
 *------------------------------------------------------------------*/
void
ComienzaNumero (void)
  {
  if ( !HaciendoNumero )
    {
    PosicionEntrada = 1;
    Entrada[0] = ' ';
    Entrada[1] = 0;
    HaciendoNumero = 1;
    }
  }

/*--------------------------------------------------------------------
 * FUNCION:  Copia()
 * OBJETIVO: Copiar la pantalla en el portapapeles
 *------------------------------------------------------------------*/
void
Copia (void)
  {
  HGLOBAL   MemoriaGlobal;
  void FAR *Memoria;

  MemoriaGlobal = GlobalAlloc (GHND, TAMPANTALLA+1);
  if ( MemoriaGlobal )
    {
    Memoria = GlobalLock (MemoriaGlobal);
    lstrcpy (Memoria, Pantalla);
    GlobalUnlock (MemoriaGlobal);

    OpenClipboard (VentanaAplicacion);
    EmptyClipboard();
    SetClipboardData (CF_TEXT, MemoriaGlobal);
    CloseClipboard();
    }
  }