/*--------------------------------------------------------------------
 * FICHERO:  ColorHTM.c
 * OBJETIVO: Mdulo principal de "Color HTML"
 * AUTOR:    Pedro Reina
 * FECHA:    L.22.2.1999
 *------------------------------------------------------------------*/

/*--------------------------------------------------------------------
 * Ficheros de cabecera
 *------------------------------------------------------------------*/

#include "ColorHTM.h"
#include <string.h>      // strcat()  strcpy()
#include <stdio.h>       // sprintf()

/*--------------------------------------------------------------------
 * Declaracin de funciones
 *------------------------------------------------------------------*/

LONG FAR PASCAL ProcedimientoPrincipal (HWND, UINT, WPARAM, LPARAM);
void BotonNormal (HWND, int);
void BotonDefecto (HWND, int);

void     CopiaAlPortapapeles (HWND, char *);
int      AjustaValor (int, int, int);
COLORREF CalculaColor (int *, char *);

/*--------------------------------------------------------------------
 * Definicin de macros constantes
 *------------------------------------------------------------------*/

// Ninguno

/*--------------------------------------------------------------------
 * Definicin de macros funciones
 *------------------------------------------------------------------*/

/*--------------------------------------------------------------------
 * MACRO:    ConfirmaSalida()
 * OBJETIVO: Pedir al usuario confirmacin para salir del programa
 *------------------------------------------------------------------*/
#define ConfirmaSalida(v) (MessageBox(v,                   \
                           "Quieres salir del programa?", \
                           "Salir", MB_ICONQUESTION | MB_OKCANCEL) == IDOK)

/*--------------------------------------------------------------------
 * MACRO:    EscribeWinIni()
 * OBJETIVO: Escribir en disco duro la versin de Win.ini que
 *           MS Windows mantiene en memoria RAM
 *------------------------------------------------------------------*/
#define EscribeWinIni()  WriteProfileString(NULL,NULL,NULL)

/*--------------------------------------------------------------------
 * MACRO:    Error()
 * OBJETIVO: Notificar un error
 * ENTRADAS: La ventana que llama y el texto del error
 *------------------------------------------------------------------*/
#define Error(v,t) (MessageBox(v,t,"Error",MB_ICONSTOP | MB_OK))

/*--------------------------------------------------------------------
 * MACRO:    AnchoPantalla()
 * OBJETIVO: Calcular el ancho total de la pantalla
 *------------------------------------------------------------------*/
#define AnchoPantalla()  GetSystemMetrics(SM_CXSCREEN)

/*--------------------------------------------------------------------
 * MACRO:    AltoPantalla()
 * OBJETIVO: Calcular el alto total de la pantalla
 *------------------------------------------------------------------*/
#define AltoPantalla()  GetSystemMetrics(SM_CYSCREEN)

/*--------------------------------------------------------------------
 * Definicin de variables globales
 *------------------------------------------------------------------*/

FARPROC PunteroAyuda;
FARPROC PunteroAcercaDe;
HANDLE  Aplic;

char ColorHTML[8];          // Tendr contenido #XXXXXX
int  Color[3] = {0, 0, 0};  // Valores de 0 a 5, RGB
COLORREF ColorTotal;        // El color elegido, en RBG

/*--------------------------------------------------------------------
 * Programa principal
 *------------------------------------------------------------------*/
#pragma argsused
int PASCAL
WinMain
  (
  HANDLE Instancia,    // Manejador de instancia de aplicacin
  HANDLE Anterior,     // Manejador de instancia anterior
  LPSTR  Parametros,   // Cadena con la lnea de parmetros
  int    ModoMostrar   // Aplicacin como ventana o como icono
  )
  {
  HWND     VentanaPrincipal, Aux;
  MSG      Mensaje;
  WNDCLASS Clase;
  HMENU    MenuControl;

  // Almacenamos la Instancia en una variable global
  Aplic = Instancia;

  // Si es necesario, registramos la clase de ventana
  if ( !Anterior )
    {
    // Estilo de ventana
    Clase.style = CS_HREDRAW | CS_VREDRAW;

    // Funcin que maneja la ventana
    Clase.lpfnWndProc = ProcedimientoPrincipal;

    // Octetos extra
    Clase.cbClsExtra = 0;

    // Octetos extra
    Clase.cbWndExtra = DLGWINDOWEXTRA;

    // Instancia de aplicacin
    Clase.hInstance = Instancia;

    // Icono al minimizar
    Clase.hIcon = LoadIcon (Instancia,"ColorHTM");

    // Cursor
    Clase.hCursor = LoadCursor (NULL,IDC_ARROW);

    // Pincel de fondo
    Clase.hbrBackground = COLOR_WINDOW + 1;

    // Recurso de men
    Clase.lpszMenuName = NULL;

    // Nombre de la clase de ventana
    Clase.lpszClassName = "ColorHTM";

    if ( !RegisterClass (&Clase) )
      { return ( FALSE ); }
    }

  // Creamos la ventana principal
  VentanaPrincipal = CreateDialog (Instancia, "Principal", 0, NULL);

  // Cambiamos el men del cuadro de control
  MenuControl = GetSystemMenu (VentanaPrincipal, FALSE);
  DeleteMenu (MenuControl, SC_SIZE,     MF_BYCOMMAND);
  DeleteMenu (MenuControl, SC_MAXIMIZE, MF_BYCOMMAND);

  // Preparamos las llamadas a los cuadros de dilogo
  PunteroAyuda = MakeProcInstance (ProcedimientoAyuda, Aplic);
  PunteroAcercaDe = MakeProcInstance (ProcedimientoAcercaDe, Aplic);

  // Mostramos la ventana
  CentraVentana (VentanaPrincipal);
  ShowWindow (VentanaPrincipal, ModoMostrar);

  // Leemos la seccin [Color HTML] del Win.ini
  Color[0] = AjustaValor (GetProfileInt ("Color HTML", "Rojo", 0), 0, 5);
  Color[1] = AjustaValor (GetProfileInt ("Color HTML", "Verde", 0), 0, 5);
  Color[2] = AjustaValor (GetProfileInt ("Color HTML", "Azul", 0), 0, 5);
  ColorTotal = CalculaColor (Color, ColorHTML);

  // Preparamos los controles
  Aux = GetDlgItem (VentanaPrincipal, ID_ROJO);
  SetScrollRange (Aux, SB_CTL, 0, 5, FALSE);
  SetScrollPos (Aux, SB_CTL, Color[0], FALSE);
  Aux = GetDlgItem (VentanaPrincipal, ID_VERDE);
  SetScrollRange (Aux, SB_CTL, 0, 5, FALSE);
  SetScrollPos (Aux, SB_CTL, Color[1], FALSE);
  Aux = GetDlgItem (VentanaPrincipal, ID_AZUL);
  SetScrollRange (Aux, SB_CTL, 0, 5, FALSE);
  SetScrollPos (Aux, SB_CTL, Color[2], FALSE);
  SetDlgItemText (VentanaPrincipal, ID_RESULTADO, ColorHTML);

  // Manejamos los mensajes
  while ( GetMessage (&Mensaje, NULL, NULL, NULL) )
    {
    TranslateMessage (&Mensaje);
    DispatchMessage (&Mensaje);
    }

  return ( Mensaje.wParam );
  }

/*--------------------------------------------------------------------
 * Definicin de funciones
 *------------------------------------------------------------------*/

/*--------------------------------------------------------------------
 * FUNCION:  ProcedimientoPrincipal()
 * OBJETIVO: Recibir los mensajes de Windows para la clase
 *           de ventanas principal de esta aplicacin
 *------------------------------------------------------------------*/
LONG FAR PASCAL
ProcedimientoPrincipal
  (
  HWND   Ventana,        // Ventana a la que se enva el mensaje
  UINT   Mensaje,        // El mensaje
  WPARAM ParametroCorto, // Informacin adicional
  LPARAM ParametroLargo  // Informacin adicional
  )
  {
  static int  Opcion = ID_COPIAR;
  static int  IdBarra = ID_ROJO;
  RECT        Muestra = {230, 10, 418, 124};
  HWND        Barra;
  int         Incremento, i, BarraAnterior;
  int         Componente;            // 0, 1 o 2: R, G o B
  BOOL        Respuesta = FALSE;
  PAINTSTRUCT Pintura;
  HDC         Contexto;
  HBRUSH      Anterior, Pincel;
  char        Aux[2];

  switch ( Mensaje )
    {
    case WM_KEYDOWN:
      BotonNormal (Ventana, Opcion);
      switch ( ParametroCorto )
        {
        case VK_TAB:
          if ( GetKeyState (VK_SHIFT) < 0 )  { Opcion--; }
          else                               { Opcion++; }
          break;
        case VK_RIGHT: Opcion++; break;
        case VK_LEFT:  Opcion--; break;
        case VK_SPACE:
        case VK_RETURN:
          SendMessage (Ventana, WM_COMMAND, Opcion, 0L);
          break;
        case VK_ESCAPE:
        case 'S': SendMessage (Ventana, WM_COMMAND, ID_SALIR, 0L);    break;
        case 'C': SendMessage (Ventana, WM_COMMAND, ID_COPIAR, 0L);   break;
        case 'Y': SendMessage (Ventana, WM_COMMAND, ID_AYUDA, 0L);    break;
        case 'D': SendMessage (Ventana, WM_COMMAND, ID_ACERCADE, 0L); break;
        case 'T': SendMessage (Ventana, WM_COMMAND, ID_AUTOR, 0L);    break;
        case 'R': IdBarra = ID_ROJO;  break;
        case 'V': IdBarra = ID_VERDE; break;
        case 'A': IdBarra = ID_AZUL;  break;
        case 'M':
        case VK_ADD:
          SendMessage (Ventana, WM_HSCROLL, SB_LINEDOWN,
            (DWORD)GetDlgItem (Ventana, IdBarra) << 16);
          break;
        case 'N':
        case VK_SUBTRACT:
          SendMessage (Ventana, WM_HSCROLL, SB_LINEUP,
            (DWORD)GetDlgItem (Ventana, IdBarra) << 16);
          break;
        case VK_HOME:
          BarraAnterior = IdBarra;
          for ( i = ID_ROJO ; i <= ID_AZUL ; i++ )
            {
            SendMessage (Ventana, WM_HSCROLL, SB_TOP,
              (DWORD)GetDlgItem (Ventana, i) << 16);
            }
          IdBarra = BarraAnterior;
          break;
        case VK_END:
          BarraAnterior = IdBarra;
          for ( i = ID_ROJO ; i <= ID_AZUL ; i++ )
            {
            SendMessage (Ventana, WM_HSCROLL, SB_BOTTOM,
              (DWORD)GetDlgItem (Ventana, i) << 16);
            }
          IdBarra = BarraAnterior;
          break;
        }
      if ( Opcion > ID_AUTOR )  { Opcion = ID_SALIR; }
      if ( Opcion < ID_SALIR )  { Opcion = ID_AUTOR; }
      BotonDefecto (Ventana, Opcion);
      break;

    case WM_HSCROLL:
      // Vemos qu barra ha sido pulsada
      Barra = HIWORD (ParametroLargo);
      if ( Barra == GetDlgItem (Ventana, ID_ROJO)  )
        {
        IdBarra = ID_ROJO;
        Componente = 0;
        }
      if ( Barra == GetDlgItem (Ventana, ID_VERDE) )
        {
        IdBarra = ID_VERDE;
        Componente = 1;
        }
      if ( Barra == GetDlgItem (Ventana, ID_AZUL)  )
        {
        IdBarra = ID_AZUL;
        Componente = 2;
        }

      // Calculamos la nueva posicin de la barra
      switch ( ParametroCorto )
        {
        case SB_LINEDOWN: Incremento =  1; break;
        case SB_LINEUP:   Incremento = -1; break;
        case SB_PAGEDOWN: Incremento =  2; break;
        case SB_PAGEUP:   Incremento = -2; break;
        case SB_BOTTOM:   Incremento =  6; break;
        case SB_TOP:      Incremento = -6; break;
        case SB_THUMBPOSITION:
        case SB_THUMBTRACK:
          Color[Componente] = LOWORD (ParametroLargo);
          Incremento = 0;
          break;
        default: Incremento = 0; break;
        }
      Color[Componente] = AjustaValor (Color[Componente]+Incremento, 0, 5);
      SetScrollPos (Barra, SB_CTL, Color[Componente], TRUE);

      // Calculamos el nuevo color
      ColorTotal = CalculaColor (Color, ColorHTML);
      SetDlgItemText (Ventana, ID_RESULTADO, ColorHTML);

      // Y pedimos que se redibuje la muestra
      InvalidateRect (Ventana, &Muestra, TRUE);
      break;

    case WM_COMMAND:
      switch ( ParametroCorto )
        {
        case ID_COPIAR:
          CopiaAlPortapapeles (Ventana, ColorHTML);
          Respuesta = TRUE;
          break;

        case ID_SALIR:
          SendMessage (Ventana, WM_CLOSE, 0, 0L);
          Respuesta = TRUE;
          break;

        case ID_AYUDA:
          MuestraAyuda (Ventana, "Ayuda", "AyudaPrincipal");
          break;

        case ID_ACERCADE:
          DialogBox (Aplic, "AcercaDe", Ventana, PunteroAcercaDe);
          Respuesta = TRUE;
          break;

        case ID_AUTOR:
          MuestraAyuda (Ventana, "Autor", "AyudaAutor");
          break;
        }
      SetFocus (Ventana);
      break;

    case WM_PAINT:
      Contexto = BeginPaint (Ventana, &Pintura);
      Pincel = CreateSolidBrush (ColorTotal);
      Anterior = SelectObject (Contexto, Pincel);
      Rectangle (Contexto, Muestra.left, Muestra.top,
                           Muestra.right, Muestra.bottom);
      SelectObject (Contexto, Anterior);
      DeleteObject (Pincel);
      EndPaint (Ventana, &Pintura);
      Respuesta = TRUE;
      break;

    case WM_CLOSE:
      if ( ConfirmaSalida (Ventana) )
        {
        sprintf (Aux, "%d", Color[0]);
        WriteProfileString ("Color HTML", "Rojo", Aux);
        sprintf (Aux, "%d", Color[1]);
        WriteProfileString ("Color HTML", "Verde", Aux);
        sprintf (Aux, "%d", Color[2]);
        WriteProfileString ("Color HTML", "Azul", Aux);
        EscribeWinIni();
        SendMessage (Ventana, WM_DESTROY, 0, 0L);
        }
      break;

    case WM_DESTROY: PostQuitMessage (0);  break;

    default:
      Respuesta = DefWindowProc (Ventana, Mensaje,
                                 ParametroCorto, ParametroLargo);
      break;
    }

  return ( Respuesta );
  }

/*--------------------------------------------------------------------
 * FUNCION:  BotonNormal()
 * OBJETIVO: Poner un boton de un cuadro de dilogo como normal
 * ENTRADAS: La ventana del cuadro y el identificador del botn
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void
BotonNormal
  (
  HWND Ventana,
  int  Opcion
  )
  {
  HWND Boton;

  Boton = GetDlgItem (Ventana, Opcion);
  SetWindowLong (Boton, GWL_STYLE, BS_PUSHBUTTON | WS_CHILD | WS_VISIBLE);
  InvalidateRect (Boton, NULL, TRUE);
  UpdateWindow (Boton);
  }

/*--------------------------------------------------------------------
 * FUNCION:  BotonDefecto()
 * OBJETIVO: Poner un boton de un cuadro de dilogo como botn por defecto
 * ENTRADAS: La ventana del cuadro y el identificador del botn
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void
BotonDefecto
  (
  HWND Ventana,
  int  Opcion
  )
  {
  HWND Boton;

  Boton = GetDlgItem (Ventana, Opcion);
  SetWindowLong (Boton, GWL_STYLE, BS_DEFPUSHBUTTON | WS_CHILD | WS_VISIBLE);
  InvalidateRect (Boton, NULL, TRUE);
  UpdateWindow (Boton);
  }

/*--------------------------------------------------------------------
 * FUNCION:  CentraVentana()
 * OBJETIVO: Centrar una ventana en pantalla
 * ENTRADAS: La ventana
 * SALIDAS:  Lgica, que indica si ha cambiado la posicin de la ventana
 * NOTA:     Slo se cambia la posicin
 *------------------------------------------------------------------*/
BOOL
CentraVentana
  (
  HWND Ventana
  )
  {
  BOOL Respuesta = FALSE;
  RECT Info;
  int  AnchoVentana, AltoVentana, Izquierda, Arriba;

  GetWindowRect (Ventana, &Info);

  AnchoVentana = Info.right - Info.left;
  AltoVentana = Info.bottom - Info.top;

  Izquierda = (AnchoPantalla()-AnchoVentana) / 2;
  Arriba = (AltoPantalla()-AltoVentana) / 2;

  if ( Izquierda >= 0 && Arriba >= 0 )
    {
    SetWindowPos (Ventana, HWND_TOP, Izquierda, Arriba, 0, 0,
                  SWP_NOACTIVATE | SWP_NOSIZE | SWP_NOREDRAW | SWP_NOZORDER);
    Respuesta = TRUE;
    }

  return ( Respuesta );
  }

/*--------------------------------------------------------------------
 * FUNCION:  CopiaAlPortapapeles()
 * OBJETIVO: Copiar un texto en el portapapeles
 * ENTRADAS: La ventana que manda el texto y el texto
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void
CopiaAlPortapapeles (HWND Ventana, char *Texto)
  {
  HGLOBAL   MemoriaGlobal;
  void FAR *Memoria;

  MemoriaGlobal = GlobalAlloc (GHND, strlen(Texto)+1);
  if ( MemoriaGlobal )
    {
    Memoria = GlobalLock (MemoriaGlobal);
    lstrcpy (Memoria, Texto);
    GlobalUnlock (MemoriaGlobal);

    OpenClipboard (Ventana);
    EmptyClipboard();
    SetClipboardData (CF_TEXT, MemoriaGlobal);
    CloseClipboard();
    }
  }

/*--------------------------------------------------------------------
 * FUNCION:  AjustaValor()
 * OBJETIVO: Poner un valor entre lmites
 * ENTRADAS: El valor, el mnimo y el mximo
 * SALIDAS:  El nuevo valor
 *------------------------------------------------------------------*/
int
AjustaValor (int Valor, int Min, int Max)
  {
  int Respuesta;

  Respuesta = Valor;
  if ( Respuesta < Min )  { Respuesta = Min; }
  if ( Respuesta > Max )  { Respuesta = Max; }

  return ( Respuesta );
  }

/*--------------------------------------------------------------------
 * FUNCION:  CalculaColor()
 * OBJETIVO: Calcular un color a partir de sus componentes de 0 a 5
 * ENTRADAS: La terna de componentes y una cadena donde dejar el color
 *           en formato HTML
 * SALIDAS:  El color RGB
 *------------------------------------------------------------------*/
COLORREF
CalculaColor (int Color[], char * ColorHTML)
  {
  int i;
  static char ValorHexa[6][3] = { "00", "33", "66", "99", "CC", "FF" };
  static int  Valor[6] = { 0x00, 0x33, 0x66, 0x99, 0xCC, 0xFF };

  strcpy (ColorHTML, "#");
  for ( i=0 ; i<3 ; i++ )
    { strcat (ColorHTML, ValorHexa[Color[i]]); }

  return ( RGB (Valor[Color[0]], Valor[Color[1]], Valor[Color[2]]) );
  }