/*--------------------------------------------------------------------
 * FICHERO:  Calculad.c
 * OBJETIVO: Definir las funciones del objeto Calculadora
 * AUTOR:    Pedro Reina
 * FECHA:    V.19.7.1996
 *------------------------------------------------------------------*/

/*--------------------------------------------------------------------
 * Ficheros de cabecera
 *------------------------------------------------------------------*/

#include "Calculad.h"

/*--------------------------------------------------------------------
 * Definicin de funciones
 *------------------------------------------------------------------*/

/*--------------------------------------------------------------------
 * FUNCION:  Cal_Crea()
 * OBJETIVO: Crear un objeto "calculad" y darle valores iniciales
 * ENTRADAS: La profundidad de la pila (entre 1 y 255)
 * SALIDAS:  Una calculad o NIL si no se puede crear
 * EJEMPLO:  Cal_Crea ( 8 )
 *------------------------------------------------------------------*/
calculad Cal_Crea (Profundidad)
octeto Profundidad;
  {
  calculad  Nueva;
  entero   *Pila;

  if ( Nueva = (calculad) (Mem_Crea (sizeof(Cal_st))) )
    {
    Cal_PonProfundidad (Nueva, Profundidad);

    if ( Pila = (entero *) Mem_Crea (Profundidad * sizeof(entero)) )
      {
      Cal_PonPila (Nueva, Pila);
      Mem_Asigna ((memoria) Pila, 0, Profundidad * sizeof(entero));
      }
    else
      {
      Cal_Destruye (Nueva);
      Nueva = NIL;
      }
    }

  else  { Nueva = NIL; }

  return ( Nueva );
  }

/*--------------------------------------------------------------------
 * FUNCION:  Cal_Destruye()
 * OBJETIVO: Destruir un objeto calculad, liberando su memoria
 * ENTRADAS: La calculad
 * SALIDAS:  Ninguna
 * EJEMPLO:  Cal_Destruye ( Calculadora )
 * ALGORITMO:
 *      Si se ha reservado memoria para Pila, liberarla
 *      Liberar la memoria del objeto calculad
 *------------------------------------------------------------------*/
void Cal_Destruye ( Calculadora )
calculad Calculadora;
  {
  if ( Cal_Pila (Calculadora) )  { Mem_Destruye (Cal_Pila(Calculadora)); }
  Mem_Destruye (Calculadora);
  }

/*-------------------------------------------------------------------------
 * FUNCION:  Cal_Inserta()
 * OBJETIVO: Insertar un nmero en una calculad
 * ENTRADAS: La calculad y el nmero
 * SALIDAS:  El nmero insertado
 * NOTA:     Se pierde el valor ms alto de la pila
 * ALGORITMO: Mover todos los niveles hacia arriba
 *            Poner el nmero en el nivel ms bajo
 *-----------------------------------------------------------------------*/
entero Cal_Inserta (Calculadora, Numero)
calculad Calculadora;
entero   Numero;
  {
  octeto i;

  for ( i=Cal_Profundidad(Calculadora)-1 ; i ; i-- )
    { Cal_PonElemento(Calculadora,i,Cal_Elemento(Calculadora,i-1)); }

  Cal_PonElemento(Calculadora, 0, Numero);

  return ( Numero );
  }

/*-------------------------------------------------------------------------
 * FUNCION:  Cal_Opera()
 * OBJETIVO: Realizar una operacin en una calculad
 * ENTRADAS: La calculadora y la operacin que hay que realizar
 *           entre 'y' (elemento 1) y 'x' (elemento 0)
 *           La operacin puede ser:
 *             CAL_SUMA -> y + x
 *             CAL_DIFERENCIA -> y - x
 *             CAL_PRODUCTO -> y * x
 *             CAL_COCIENTE -> y / x (si x != 0)
 *                             Nada  (si x == 0)
 *             CAL_ALEATORIO -> Un nmero entre 'y' y 'x'
 * SALIDAS:   El resultado de la operacin
 * NOTAS:     1. Se duplica el valor ms alto de la pila
 *            2. Los resultados siempre son enteros
 * ALGORITMO: Realizar la operacin entre los niveles ms bajos
 *            Escribir el resultado en el nivel ms bajo
 *            Bajar los niveles
 *-----------------------------------------------------------------------*/
entero Cal_Opera (Calculadora, Operacion)
calculad Calculadora;
octeto   Operacion;
  {
  octeto i;
  entero Respuesta;

  if ( Operacion == CAL_COCIENTE && !Cal_Resultado(Calculadora) )
    { Respuesta = 0; }

  else
    {
    switch ( Operacion )
      {
      case CAL_SUMA:
        Respuesta = Cal_Elemento(Calculadora,1)+Cal_Elemento(Calculadora,0);
        break;
      case CAL_DIFERENCIA:
        Respuesta = Cal_Elemento(Calculadora,1)-Cal_Elemento(Calculadora,0);
        break;
      case CAL_PRODUCTO:
        Respuesta = Cal_Elemento(Calculadora,1)*Cal_Elemento(Calculadora,0);
        break;
      case CAL_COCIENTE:
        Respuesta = Cal_Elemento(Calculadora,1)/Cal_Elemento(Calculadora,0);
        break;
      case CAL_ALEATORIO:
        Respuesta = Azr_Entero(Cal_Elemento(Calculadora,1),
                               Cal_Elemento(Calculadora,0));
        break;
      }
    }

  Cal_PonElemento (Calculadora, 0, Respuesta);

  for ( i=1 ; i<Cal_Profundidad(Calculadora)-1 ; i++ )
    { Cal_PonElemento (Calculadora, i, Cal_Elemento(Calculadora,i+1)); }

  return ( Respuesta );
  }

/*-------------------------------------------------------------------------
 * FUNCION:  Cal_Aplica()
 * OBJETIVO: Aplicar una funcin en una calculad
 * ENTRADAS: La calculadora y la funcin que hay que aplicar
 *           al elemento 0
 *           La operacin puede ser:
 *             CAL_CUADRADO
 *             CAL_CUBO
 * SALIDAS:   El resultado de la funcin
 *-----------------------------------------------------------------------*/
entero Cal_Aplica (Calculadora, Funcion)
calculad Calculadora;
octeto   Funcion;
  {
  entero Respuesta, Valor;

  Valor = Cal_Elemento (Calculadora,0);

  switch ( Funcion )
    {
    case CAL_CUADRADO:
      Respuesta = Valor * Valor;
      break;
    case CAL_CUBO:
      Respuesta = Valor * Valor * Valor;
      break;
    }

  Cal_PonElemento (Calculadora, 0, Respuesta);

  return ( Respuesta );
  }

/*-------------------------------------------------------------------------
 * FUNCION:  Cal_Intercambia()
 * OBJETIVO: Intercambiar los valores de los elementos 0 y 1
 *           de una calculad
 * ENTRADAS: La calculadora
 * SALIDAS:  El resultado de la operacin
 *-----------------------------------------------------------------------*/
entero Cal_Intercambia (Calculadora)
calculad Calculadora;
  {
  entero Aux;

  Aux = Cal_Elemento (Calculadora,1);

  Cal_PonElemento (Calculadora, 1, Cal_Elemento(Calculadora,0));
  Cal_PonElemento (Calculadora, 0, Aux);

  return ( Aux );
  }

/*-------------------------------------------------------------------------
 * FUNCION:  Cal_Ejecuta()
 * OBJETIVO: Ejecutar un programa en una calculadora
 * ENTRADAS: La calculad y el programa. ste es una porcin
 *           de memoria con cdigos que dirigen la actuacin
 * SALIDAS:  El resultado que quede en la calculadora
 *-----------------------------------------------------------------------*/
entero Cal_Ejecuta (Calculadora, Programa)
calculad Calculadora;
memoria  Programa;
  {
  entero   i, Numero;
  menudo   Signo;
  logico   Sigue=SI;

  for ( i=0 ; Sigue ; i++ )
    {
    switch ( Programa[i] )
      {
      case CAL_FIN_PROGRAMA: Sigue = NO; break;
      case CAL_INTERCAMBIA:
        Cal_Intercambia (Calculadora);
        break;
      case CAL_DUPLICA:
        Cal_Duplica (Calculadora);
        break;
      case CAL_OPERACION:
        Cal_Opera (Calculadora, Programa[++i]);
        break;
      case CAL_FUNCION:
        Cal_Aplica (Calculadora, Programa[++i]);
        break;
      case CAL_NUMERO_POS:
      case CAL_NUMERO_NEG:
        Signo = Programa[i] == CAL_NUMERO_POS ? 1 : -1;
        Numero = 256 * Programa[++i];
        Numero += Programa[++i];
        Numero *= Signo;
        Cal_Inserta (Calculadora, Numero);
        break;
      }
    }

  return ( Cal_Resultado(Calculadora) );
  }
