/*--------------------------------------------------------------------
 * FICHERO:  Dibugris.c
 * OBJETIVO: Definir las funciones Dgr_Crea(), Dgr_Destruye(),
 *           Dgr_PonPunto() y Dgr_Invierte()
 * AUTOR:    Pedro Reina
 * FECHA:    V.19.7.1996
 *------------------------------------------------------------------*/

/*--------------------------------------------------------------------
 * Ficheros de cabecera
 *------------------------------------------------------------------*/

#include "Dibugris.h"

/*--------------------------------------------------------------------
 * Definicin de funciones
 *------------------------------------------------------------------*/

/*--------------------------------------------------------------------
 * FUNCION:  Dgr_Crea()
 * OBJETIVO: Crear un objeto "dibugris" y darle valores iniciales
 * ENTRADAS: El ancho y el alto requeridos
 * SALIDAS:  Un dibugris o NIL si no se puede crear
 * EJEMPLO:  Dgr_Crea (80,30)
 *------------------------------------------------------------------*/
dibugris Dgr_Crea (Ancho, Alto)
entero Ancho, Alto;
  {
  dibugris Nuevo;
  memoria  Memoria;

  if ( Nuevo = (dibugris) (Mem_Crea (sizeof(Dgr_st))) )
    {
    Mem_Asigna (Nuevo, 0, sizeof(Dgr_st));  /* Puesta a cero de todo */

    Dgr_PonAncho (Nuevo, Ancho);
    Dgr_PonAlto (Nuevo, Alto);
    Dgr_PonAnchoEnOxel (Nuevo, (Ancho+1)/2);
    Dgr_PonTamano (Nuevo, Dgr_AnchoEnOxel(Nuevo) * Alto);

    if ( Memoria = Mem_Crea (Dgr_Tamano(Nuevo)) )
      {
      Dgr_PonDato (Nuevo, Memoria);
      Mem_Asigna (Memoria, 0, Dgr_Tamano(Nuevo));
      }
    else
      {
      Dgr_Destruye (Nuevo);
      Nuevo = NIL;
      }
    }

  else { Nuevo = NIL; }

  return ( Nuevo );
  }

/*--------------------------------------------------------------------
 * FUNCION:  Dgr_Destruye()
 * OBJETIVO: Destruir un objeto dibugris, liberando su memoria
 * ENTRADAS: El dibugris
 * SALIDAS:  Ninguna
 * EJEMPLO:  Dgr_Destruye ( Dibujo )
 * ALGORITMO:
 *      Si se ha reservado memoria para Dato, liberarla
 *      Liberar la memoria del objeto dibugris
 *------------------------------------------------------------------*/
void Dgr_Destruye ( Dibujo )
diblane Dibujo;
  {
  if ( Dgr_Dato (Dibujo) )  { Mem_Destruye (Dgr_Dato(Dibujo)); }
  Mem_Destruye (Dibujo);
  }

/*--------------------------------------------------------------------
 * FUNCION:  Dgr_Invierte()
 * OBJETIVO: Invertir los colores de un objeto dibugris
 * ENTRADAS: Un dibugris
 * SALIDAS:  Ninguna
 * EJEMPLO:  Dgr_Invierte (Dibujo)
 *------------------------------------------------------------------*/
void Dgr_Invierte (Dibujo)
dibugris Dibujo;
  {
  entero Fila, Columna;

  for (  Fila=0 ; Fila < Dgr_Alto(Dibujo) ; Fila++ )
    {
    for (  Columna=0 ; Columna < Dgr_Ancho(Dibujo) ; Columna++ )
      { Dgr_PonPunto (Dibujo, Fila, Columna,
                      15-Dgr_Punto(Dibujo, Fila, Columna));
      }
    }
  }

/*--------------------------------------------------------------------
 * FUNCION:  Dgr_PonPunto()
 * OBJETIVO: Poner el valor de un punto de un dibugris
 * ENTRADAS: El dibugris, la fila y la columna del punto y el
 *           nuevo valor
 * SALIDAS:  El nuevo valor del punto, que est entre 0 y 15
 * NOTAS:    1. La fila 0 es la de arriba
 *           2. La columna 0 es la de la izquierda
 *           3. Se asume que el punto pertenece al dibugris
 * EJEMPLO:  Dgr_PonPunto ( Dibujo, 4, 17, 7 )
 *------------------------------------------------------------------*/
octeto Dgr_PonPunto (Imagen, Fila, Columna, Valor)
dibugris Imagen;
entero   Fila, Columna;
octeto   Valor;
  {
  entero Pos;
  octeto Aux;

  Pos = Fila * Dgr_AnchoEnOxel(Imagen) + Columna / 2;
  Aux = Dgr_Dato(Imagen)[Pos];

  if ( Columna % 2 )
    {
    Aux &= 0xF0;
    Aux |= Valor;
    }
  else
    {
    Aux &= 0x0F;
    Aux |= Valor << 4;
    }

  Dgr_Dato(Imagen)[Pos] = Aux;

  return ( Valor );
  }
