/*--------------------------------------------------------------------
 * FICHERO:       DibuGris.h
 * OBJETIVO:      Definir el objeto "Dibugris" (Dibujo en gama de grises)
 * IDENTIFICADOR: Dgr
 * AUTOR:         Pedro Reina
 * FECHA:         D.29.3.1998
 *------------------------------------------------------------------*/

/*--------------------------------------------------------------------
 * Funciones pblicas
 *
 *   Dgr_Alto()              Dibugris.h
 *   Dgr_Ancho()             Dibugris.h
 *   Dgr_AnchoEnOxel()       Dibugris.h
 *   Dgr_Borra()             Dibugris.h
 *   Dgr_Crea()              Dibugris.c
 *   Dgr_Dato()              Dibugris.h
 *   Dgr_Destruye()          Dibugris.c
 *   Dgr_Invierte()          Dibugris.c
 *   Dgr_PonPunto()          Dibugris.h
 *   Dgr_Punto()             Dibugris.h
 *   Dgr_Tamano()            Dibugris.h
 *-----------------------------------------------------------------*/

/*--------------------------------------------------------------------
 * Funciones conocidas
 *
 *   Ninguna
 *------------------------------------------------------------------*/

/*--------------------------------------------------------------------
 * Funciones privadas
 *
 *   Bdn_PonDato()           Dibugris.h
 *   Dgr_PonAlto()           Dibugris.h
 *   Dgr_PonAncho()          Dibugris.h
 *   Dgr_PonAnchoEnOxel()    Dibugris.h
 *   Dgr_PonTamano()         Dibugris.h
 *------------------------------------------------------------------*/

#ifndef _DIBUGRIS_
#define _DIBUGRIS_

/*--------------------------------------------------------------------
 * Ficheros de cabecera
 *------------------------------------------------------------------*/

#include <Olimpo.h>  /* Versin 2.0 */

/*--------------------------------------------------------------------
 * Definicin de macros constantes
 *------------------------------------------------------------------*/

    /* Ninguno */

/*--------------------------------------------------------------------
 * Definicin del tipo "dibugris"
 *------------------------------------------------------------------*/
typedef struct {
               entero  Ancho;
               entero  Alto;
               entero  AnchoEnOxel;
               entero  Tamano;
               memoria Dato;
               } Dgr_st, *dibugris;

/*--------------------------------------------------------------------
 * Variables globales
 *------------------------------------------------------------------*/

   /* Ninguna */

/*--------------------------------------------------------------------
 * Declaracin de funciones
 *------------------------------------------------------------------*/

dibugris Dgr_Crea();
void     Dgr_Destruye();
void     Dgr_Invierte();
octeto   Dgr_PonPunto();

/*--------------------------------------------------------------------
 * Estructura de los datos en un dibugris
 *
 * La funcin Dgr_Dato() da la direccin en memoria donde est almacenado
 * el conjunto de datos que definen un dibujo en gama de grises
 *
 * Los puntos del dibujo estn representados por grupos de 4 bits, de modo
 * que cada punto puede tener un valor de 0 a 15. El valor 0 se considera
 * color negro y el valor 15 color blanco. Los dems nmeros representan
 * tonos de gris.
 *
 * Los valores de los puntos estn agrupados de 2 en 2 en posiciones en
 * memoria que se consideran como octetos, es decir, valores sin signo.
 * Cada grupo de 2 puntos se denomina "oxel".
 *
 * Los valores se van escribiendo en memoria de arriba hacia abajo por filas
 * y dentro de cada fila de izquierda a derecha. Si los puntos de cada fila
 * no son mltiplo de 2, el ltimo oxel se rellena hasta completar el octeto
 * mediante bits que deben estar a 0.
 *------------------------------------------------------------------*/

/*--------------------------------------------------------------------
 * Definicin de macros funciones
 *------------------------------------------------------------------*/

/*--------------------------------------------------------------------
 * MACRO:    Dgr_Ancho()
 * OBJETIVO: Decir el ancho en pixels de un dibugris
 * ENTRADAS: El dibugris
 * SALIDAS:  El ancho en pixels
 * EJEMPLO:  Dgr_Ancho ( Dibujo )
 *------------------------------------------------------------------*/
#define Dgr_Ancho(d)  ((d)->Ancho)

/*--------------------------------------------------------------------
 * MACRO:    Dgr_PonAncho()
 * OBJETIVO: Poner el ancho en pixels de un dibugris
 * ENTRADAS: El dibugris y el ancho en pixels
 * SALIDAS:  El ancho en pixels
 * EJEMPLO:  Dgr_PonAncho ( Dibujo, 24 )
 *------------------------------------------------------------------*/
#define Dgr_PonAncho(d,a)  (((d)->Ancho)=(a))

/*--------------------------------------------------------------------
 * MACRO:    Dgr_Alto()
 * OBJETIVO: Decir el alto en pixels de un dibugris
 * ENTRADAS: El dibugris
 * SALIDAS:  El alto en pixels
 * EJEMPLO:  Dgr_Alto ( Dibujo )
 *------------------------------------------------------------------*/
#define Dgr_Alto(d)  ((d)->Alto)

/*--------------------------------------------------------------------
 * MACRO:    Dgr_PonAlto()
 * OBJETIVO: Poner el alto en pixels de un dibugris
 * ENTRADAS: El dibugris y el alto en pixels
 * SALIDAS:  El alto en pixels
 * EJEMPLO:  Dgr_PonAlto ( Dibujo, 12 )
 *------------------------------------------------------------------*/
#define Dgr_PonAlto(d,a)  (((d)->Alto)=(a))

/*--------------------------------------------------------------------
 * MACRO:    Dgr_AnchoEnOxel()
 * OBJETIVO: Decir el ancho en oxels de un dibugris
 * ENTRADAS: El dibugris
 * SALIDAS:  El ancho en oxels
 * EJEMPLO:  Dgr_AnchoEnOxel ( Dibujo )
 *------------------------------------------------------------------*/
#define Dgr_AnchoEnOxel(d)  ((d)->AnchoEnOxel)

/*--------------------------------------------------------------------
 * MACRO:    Dgr_PonAnchoEnOxel()
 * OBJETIVO: Poner el ancho en oxels de un dibugris
 * ENTRADAS: El dibugris y el ancho en oxels
 * SALIDAS:  El ancho en oxels
 * EJEMPLO:  Dgr_PonAnchoEnOxel ( Dibujo, 4 )
 *------------------------------------------------------------------*/
#define Dgr_PonAnchoEnOxel(d,a)  (((d)->AnchoEnOxel)=(a))

/*--------------------------------------------------------------------
 * MACRO:    Dgr_Tamano()
 * OBJETIVO: Decir el tamao en octetos de un dibugris
 * ENTRADAS: El dibugris
 * SALIDAS:  El tamao en octetos
 * EJEMPLO:  Dgr_Tamano ( Dibujo )
 *------------------------------------------------------------------*/
#define Dgr_Tamano(d)  ((d)->Tamano)

/*--------------------------------------------------------------------
 * MACRO:    Dgr_PonTamano()
 * OBJETIVO: Poner el tamao en octetos de un dibugris
 * ENTRADAS: El dibugris y el tamao en octetos
 * SALIDAS:  El tamao en octetos
 * EJEMPLO:  Dgr_PonTamano ( Dibujo, 1024 )
 *------------------------------------------------------------------*/
#define Dgr_PonTamano(d,t)  (((d)->Tamano)=(t))

/*--------------------------------------------------------------------
 * MACRO:    Dgr_Dato()
 * OBJETIVO: Decir la direccin donde se encuentran los datos
 *           de un dibugris
 * ENTRADAS: El dibugris
 * SALIDAS:  La direccin en memoria de los datos
 * EJEMPLO:  Dgr_Dato ( Dibujo )
 *------------------------------------------------------------------*/
#define Dgr_Dato(d)  ((d)->Dato)

/*--------------------------------------------------------------------
 * MACRO:    Dgr_PonDato()
 * OBJETIVO: Poner la direccin donde se encuentran los datos de un dibugris
 * ENTRADAS: El dibugris y la direccin en memoria
 * SALIDAS:  La direccin en memoria de los datos
 * EJEMPLO:  Dgr_PonDato ( Dibujo, 1024 )
 *------------------------------------------------------------------*/
#define Dgr_PonDato(d,t)  (((d)->Dato)=(t))

/*--------------------------------------------------------------------
 * MACRO:    Dgr_Punto()
 * OBJETIVO: Decir el valor de un punto de un diblane
 * ENTRADAS: El dibugris, la fila y la columna del punto
 * SALIDAS:  El valor del punto, que est entre 0 y 15
 * NOTAS:    1. La fila 0 es la de arriba
 *           2. La columna 0 es la de la izquierda
 *           3. Se asume que el punto pertenece al dibugris
 * EJEMPLO:  Dgr_Punto ( Dibujo, 4, 17 )
 *------------------------------------------------------------------*/
#define Dgr_Punto(d,f,c)  ((Dgr_Dato(d)[(f)*Dgr_AnchoEnOxel(d)+(c)/2] & \
                           0xF0 >> 4*((c)%2)) >> 4*(((c)+1)%2))

/*--------------------------------------------------------------------
 * MACRO:    Dgr_Borra()
 * OBJETIVO: Borrar la imagen de un dibugris
 * ENTRADAS: El dibugris
 * SALIDAS:  Ninguna
 * EJEMPLO:  Dgr_Borra ( Dibujo )
 *------------------------------------------------------------------*/
#define Dgr_Borra(d)  (Mem_Asigna(Dgr_Dato(d),0,Dgr_Tamano(d)))

#endif /* _DIBUGRIS_ */
