/*--------------------------------------------------------------------
 * FICHERO:  Kandi.c
 * OBJETIVO: Crear ficheros TIFF de 16 niveles de gris usando frmulas
 * AUTOR:    Pedro Reina
 * FECHA:    X.1.4.1998
 *------------------------------------------------------------------*/

/*--------------------------------------------------------------------
 * Ficheros de cabecera
 *------------------------------------------------------------------*/

#include <Olimpo.h>   /*  Versin 2.0  */
#include <string.h>   /*  strtok()     */

#include "Dibugris.h"
#include "Tiff_G16.h"
#include "Calculad.h"

/*--------------------------------------------------------------------
 * Declaracin de funciones
 *------------------------------------------------------------------*/

void LeeConfig();

void Explica();
void ExplicaObjetivo();
void ExplicaManejo();
void ExplicaAutor();
void ExplicaNombre();

void   Fichero();
void   LeeFichero();
logico ExaminaTiff();
logico LeeBandas();
void   GrabaFichero();
logico GrabaBandas();

cadena NombreCompleto();
cadena EligeNombre();
int    ComparaCadena();
cadena PideNombre();
cadena PideDirectorio();
cadena PideMascara();
cadena PideExtension();

void    ManejoImagen();
cadena  PideFormula();
memoria ConvierteCadenaEnPrograma();
void    GeneraImagen();
void    MuestraImagen();
void    PideTamano();
void    PideZona();

logico CreaBandas();
void   InvierteBandas();
void   BorraBandas();

void Opciones();

void PreparaColor();
void AsignaPaletaDAC();
void CambiaColorDAC();
void RestauraColor();

/*--------------------------------------------------------------------
 * Definicin de macros constantes
 *------------------------------------------------------------------*/

#define LONG_DIRECTORIO  24
#define LONG_EXTENSION    3
#define LONG_NOMBRE      14
#define LONG_MASCARA      8

#define LONG_FORMULA     34

#define MAX_TAMANO_BANDA 8192  /* Cada banda tiene como mximo 8K      */
#define MAX_BANDA          19  /* = DivEx (640*480/2,MAX_TAMANO_BANDA) */

/*--------------------------------------------------------------------
 * Variables globales
 *------------------------------------------------------------------*/

dibugris Banda[MAX_BANDA+1];   /* La extra, para el NIL */
contpos  AltoTotal;
contador FilSup, FilInf, ColIzq, ColDer;
logico   HayQueOrdenar = NO;

/*--------------------------------------------------------------------
 * Definicin de macros funciones
 *------------------------------------------------------------------*/

#define LimpiaZonaTrabajo()  Pan_Borra(NEGRO,4,0,21,79)
#define LimpiaZonaFichero()  Pan_Borra(NEGRO,11,0,21,79)
#define LimpiaZonaImagen()   Pan_Borra(NEGRO,13,0,21,79)

/*--------------------------------------------------------------------
 * MACRO:    BorraPantallaVGA()
 * OBJETIVO: Borrar la pantalla VGA en modo grfico completa, no slo
 *           la parte que usa Olimpo
 * ENTRADAS: Ninguna
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
#define BorraPantallaVGA()  Pan_Borra(NEGRO,-3,0,27,79)

/*--------------------------------------------------------------------
 * Programa principal
 *------------------------------------------------------------------*/
void main (Narg, Arg)
int   Narg;
char *Arg[];
  {
  enum { SALIDA_ESC, SALIDA, EXPLICA, FICHERO, IMAGEN, OPCIONES };

  static cadena Principal[] = {">Salida", "E>xplica", ">Fichero",
                               ">Imagen", ">Opciones", NIL };
  cadena Nombre;
  entero Opcion=IMAGEN;
  logico Sigue=SI;

  Pan_Define (PAN_GRAFICO);
  Prg_Presenta ( "Kandi", "3.1", "Pedro Reina", "Abril 1998" );

  if ( Pan_Modo() == PAN_TEXTO )
    { Usr_ErrorFatal ("Este ordenador no dispone de grficos"); }

  if ( Narg == 1 )
    {
    Nombre = Fch_Nombre ( "Kandi","cnf");
    if ( Fch_Existe (Nombre) )
      { LeeConfig (Nombre); }
    }
  else
    {
    Nombre = Cad_Duplica ( Arg[1] );
    LeeConfig (Nombre);
    }
  Cad_Destruye (Nombre);

  CreaBandas (320, 240);

  Cdr_Caja (CDR_SIMPLE,1,0,3,79,NEGRO,VERDE);

  while ( Sigue )
    {
    Opcion = Men_Horizontal (2,2,77,Principal,Opcion);
    switch ( Opcion )
      {
      case SALIDA_ESC:
      case SALIDA:
           if ( Usr_Consulta ("Quieres terminar el programa?") )
             { Sigue = NO; }
           Opcion = SALIDA;
           break;
      case EXPLICA:       Explica();       break;
      case FICHERO:       Fichero();       break;
      case IMAGEN:        ManejoImagen();  break;
      case OPCIONES:      Opciones();      break;
      }
    LimpiaZonaTrabajo();
    }

  Pan_Cierra();
  }

/*--------------------------------------------------------------------
 * Definicin de funciones
 *------------------------------------------------------------------*/

/*--------------------------------------------------------------------
 * FUNCION:  LeeConfig()
 * OBJETIVO: Leer el fichero de configuracion
 *------------------------------------------------------------------*/
void LeeConfig (Nombre)
cadena Nombre;
  {
  config    Config;
  cadena    Linea, Informe, Error;
  contador  Tipo;
  caracter *Etiqueta, *Aux, NumeroLinea[80];

  Informe = "Leyendo configuracin";
  Usr_Informa (Informe);

  if ( Config = Cnf_Abre (Nombre) )
    {
    while ( Linea = Cnf_Lee (Config) )
      {
      if ( Etiqueta = strtok (Linea, " ") )
        {

        if ( Cad_Igual (Etiqueta,"Sonido") )
          {
          if ( Aux = strtok (NULL, " ") )
            {
            if ( toupper(Aux[0]) == 'S' ) { Son_Enciende(); }
            else                          { Son_Apaga(); }
            }
          }

        else if ( Cad_Igual (Etiqueta,"Ordenar") )
          {
          if ( Aux = strtok (NULL, " ") )
            { HayQueOrdenar = toupper(Aux[0]) == 'S'; }
          }

        else
          {
          Error = Cad_Crea (80);
          sprintf (NumeroLinea,
            "Opcin no reconocida en la lnea %d del fichero ",
            Cnf_Linea (Config));
          Error = Cad_Une (NumeroLinea, Nombre, CAD_FIN);
          Usr_Avisa (Error);
          Cad_Destruye (Error);
          Usr_Informa (Informe);
          }

        } /* Fin if Etiqueta */
      Cad_Destruye (Linea);
      } /* Fin while Linea */
    } /* Fin if Cnf_Abre() */
  }

/*--------------------------------------------------------------------
 * FUNCION:  Explica()
 * OBJETIVO: Dirigir las explicaciones
 *------------------------------------------------------------------*/
void Explica()
  {
  enum { SALIDA_ESC, OBJETIVO, MANEJO, AUTOR, NOMBRE };
  static octeto MaxOpcion = 4;

  static cadena MenuExplica[] = { ">Objetivo del programa",
                                  ">Manejo de las opciones",
                                  ">Contactar con el autor",
                                  ">Por qu se llama as",
                                  NIL };

  static cadena Presentacion[] = {
    "Puedes pedir explicacin sobre diferentes aspectos",
    "de este programa. Elige el que desees y pulsa ESC",
    "cuando hayas terminado.",
    NIL };

  logico Sigue = SI;
  entero Opcion = 1, i;

  while ( Sigue )
    {
    LimpiaZonaTrabajo();
    Pan_Color (NEGRO,BLANCO);
    for ( i=0 ; Presentacion[i] ; i++ )
      { Pan_PonTexto (6+i,15,Presentacion[i]); }

    Opcion = Men_Vertical (11,29,14,52,MenuExplica,Opcion);
    LimpiaZonaTrabajo();
    switch ( Opcion )
      {
      case SALIDA_ESC:  Sigue = NO;         break;
      case OBJETIVO:    ExplicaObjetivo();  break;
      case MANEJO:      ExplicaManejo();    break;
      case AUTOR:       ExplicaAutor();     break;
      case NOMBRE:      ExplicaNombre();    break;
      }
    if ( Opcion )  { Usr_PulsaUnaTecla(""); }
    Opcion++;
    if ( Opcion > MaxOpcion )  { Opcion = 1; }
    }
  }

/*--------------------------------------------------------------------
 * FUNCION:  ExplicaObjetivo()
 * OBJETIVO: Explicar el objetivo del programa
 *------------------------------------------------------------------*/
void ExplicaObjetivo()
  {
  static cadena Mensaje[] = {
      "Este programa permite crear imgenes de 16 niveles de gris",
      "utilizando frmulas matemticas para decidir el color de cada",
      "punto.",
      "",
      "Las imgenes creadas se pueden grabar en formato TIFF, y tambin",
      "es posible leer un fichero TIFF de 16 niveles de gris para modi-",
      "ficar su aspecto.",
      NIL };

  octeto i;

  Pan_Color (NEGRO,BLANCO);
  for ( i=0 ; Mensaje[i] ; i++ )
    { Pan_PonTexto (6+i,6,Mensaje[i]); }
  }

/*--------------------------------------------------------------------
 * FUNCION:  ExplicaManejo()
 * OBJETIVO: Explicar cmo se maneja el programa
 *------------------------------------------------------------------*/
void ExplicaManejo()
  {
  static cadena Mensaje[] = {
    "La opcin Fichero permite leer y grabar ficheros TIFF.",
    "",
    "La opcin Imagen meneja la generacin de imgenes mediante frmulas.",
    "Cuando se introduce una frmula, sta actuar sobre una calculadora",
    "con RPN, tipo HP, en la que ya estn introducidas la columna y la",
    "fila del punto que se va a calcular. El valor resultante de la fr-",
    "mula se utiliza para asignar un nivel de gris de 0 a 15 tomando m-",
    "dulo 16. Slo se aplicar el valor a los puntos de la zona que est",
    "definida.",
    "",
    "En Opciones se pueden definir dos cosas: si el sonido est encendido",
    "o apagado, y si se desea ver por orden alfabtico los nombres de los",
    "ficheros cuando se va a leer uno.",
    "",
    "Para ms informacin, lee el fichero TXT que acompaa al programa.",
    NIL };

  octeto i;

  Pan_Color (NEGRO,BLANCO);
  for ( i=0 ; Mensaje[i] ; i++ )
    { Pan_PonTexto (5+i,4,Mensaje[i]); }
  }

/*--------------------------------------------------------------------
 * FUNCION:  ExplicaAutor()
 * OBJETIVO: Explicar quin es el autor y cmo contactar con l
 *------------------------------------------------------------------*/
void ExplicaAutor()
  {
  static cadena Mensaje[] = {
    "Este programa es de Dominio Pblico. Lo puedes usar, distribuir",
    "y modificar como desees.",
    "",
    "Ha sido escrito en C usando el sistema de programacin Olimpo;",
    "si no dispones de la versin 2.0 de Olimpo, puedes cargarla",
    "directamente desde mi sede web o pedrmela si no dispones de",
    "acceso a Internet.",
    "",
    "Si tienes cualquier duda o consulta, cuntamela e intentar ayudarte.",
    "",
    "Mis datos:",
    "",
    "Pedro Reina",
    "c/ Marquesa de Argeso, 4      Telfono: 91 565 17 59",
    "28019 Madrid                   Correo electrnico: pedro@anit.es",
    "Espaa                         Web: www.anit.es/pedro",
    NIL };

  octeto i;

  Pan_Color (NEGRO,BLANCO);
  for ( i=0 ; Mensaje[i] ; i++ )
    { Pan_PonTexto (5+i,5,Mensaje[i]); }
  }

/*--------------------------------------------------------------------
 * FUNCION:  ExplicaNombre()
 * OBJETIVO: Explicar por qu se llama as el programa
 *------------------------------------------------------------------*/
void ExplicaNombre()
  {
  static cadena Mensaje[] = {
    "Este programa se llama Kandi en honor del pintor Vasili Kandinsky,",
    "que naci en Rusia en 1866 y muri en 1944. Vivi en Munich y",
    "Pars, y lider el grupo artstico conocido como \"El Jinete Azul\".",
    "",
    "Son muy importantes en su obra las series llamadas composiciones",
    "e improvisaciones.",
      NIL };

  octeto i;

  Pan_Color (NEGRO,BLANCO);
  for ( i=0 ; Mensaje[i] ; i++ )
    { Pan_PonTexto (5+i,4,Mensaje[i]); }
  }

/*--------------------------------------------------------------------
 * FUNCION:  Fichero()
 * OBJETIVO: Presentar y manejar las opciones de manejo de ficheros
 * ENTRADAS: Ninguna
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void Fichero()
  {
  enum { SALIDA_ESC, LEER, GRABAR, DIRECTORIO, MASCARA, EXTENSION };
  static cadena MenuFich[] = {">Leer", ">Grabar", ">Directorio",
                              ">Mscara", ">Extensin", NIL };

  static char Directorio[LONG_DIRECTORIO+1] = "";
  static char Nombre[LONG_NOMBRE+1]         = "";
  static char Mascara[LONG_MASCARA+1]       = "*";
  static char Extension[LONG_EXTENSION+1]   = "tif";

  cadena Aux;
  region Region;
  entero Opcion = LEER;
  logico Sigue = SI, Refresca = SI;

  if ( Region = Reg_Crea (4,20,10,61) )
    {
    Reg_Color (Region,NEGRO,VERDE);
    while ( Sigue )
      {
      if ( Refresca )
        {
        Reg_Borde (Region,CDR_SIMPLE,NEGRO,VERDE);
        Pan_Borra (NEGRO,Reg_FilSup(Region)+1, Reg_ColIzq(Region)+15,
                         Reg_FilInf(Region)-1, Reg_ColDer(Region)-1);
        Reg_PonTexto (Region,3,15,Directorio);
        Reg_PonTexto (Region,4,15,Mascara);
        Reg_PonTexto (Region,5,15,Extension);
        Refresca = NO;
        }

      Opcion = Men_Vertical (Reg_FilSup(Region)+1, Reg_ColIzq(Region)+1,
                             Reg_FilInf(Region)-1, Reg_ColIzq(Region)+14,
                             MenuFich, Opcion);
      switch ( Opcion )
        {
        case SALIDA_ESC:  Sigue = NO;      break;

        case LEER:        Aux = EligeNombre(Directorio,Mascara,Extension);
                          if ( Tec_Ultima() == TEC_ENTER )
                            {
                            if ( Cad_Longitud(Aux) > LONG_NOMBRE )
                              { Aux[LONG_NOMBRE] = NULO; }
                            Cad_Copia (Nombre,Aux);
                            LeeFichero (Directorio,Nombre,Extension);
                            Refresca = SI;
                            }
                          Cad_Destruye (Aux);
                          break;

        case GRABAR:      Aux = PideNombre(Nombre);
                          if ( Tec_Ultima() != TEC_ESC )
                            {
                            Cad_Copia (Nombre,Aux);
                            GrabaFichero (Directorio,Nombre,Extension);
                            }
                          Cad_Destruye (Aux);
                          break;

        case DIRECTORIO:  Aux = PideDirectorio(Directorio);
                          if ( Tec_Ultima() != TEC_ESC )
                            {
                            Cad_Copia (Directorio,Aux);
                            Refresca = SI;
                            }
                          Cad_Destruye (Aux);
                          break;

        case MASCARA:     Aux = PideMascara(Mascara);
                          if ( Tec_Ultima() != TEC_ESC )
                            {
                            Cad_Copia (Mascara,Aux);
                            Refresca = SI;
                            }
                          Cad_Destruye (Aux);
                          break;

        case EXTENSION:   Aux = PideExtension(Extension);
                          if ( Tec_Ultima() != TEC_ESC )
                            {
                            Cad_Copia (Extension,Aux);
                            Refresca = SI;
                            }
                          Cad_Destruye (Aux);
                          break;
        }
      LimpiaZonaFichero();
      }
    Reg_Destruye (Region);
    }
  else  { Usr_Avisa ("Falta memoria"); }
  }

/*--------------------------------------------------------------------
 * FUNCION:  LeeFichero()
 * OBJETIVO: Leer un fichero TIFF
 * ENTRADAS: Directorio, nombre y extensin del fichero TIFF
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void LeeFichero (Dir,Nom,Ext)
cadena Dir,Nom,Ext;
  {
  fichero  Fichero;
  cadena   Completo;
  tiff     Tiff;
  logico   TodoBien;

  Completo = NombreCompleto(Dir,Nom,Ext);
  if ( Tiff = Tif_Abre (Completo) )
    {
    LimpiaZonaTrabajo();
    Usr_Informa ("Leyendo fichero");
    Pan_Color (NEGRO, VERDE);
    Pan_PonTexto (5,3,"Fichero: ");
    Pan_Tinta (BLANCO);
    Pan_Texto (Completo);

    if ( TodoBien = ExaminaTiff (Tiff) )
      {
      CreaBandas ((contpos)Tif_Ancho (Tiff), (contpos)Tif_Alto (Tiff));
      TodoBien = LeeBandas (Tiff);
      if ( Tif_Interpretacion (Tiff) == TIF_BLANCO_CERO )
        { InvierteBandas(); }
      }

    Tif_Cierra (Tiff);
    }

  if ( TodoBien )
    { Usr_PulsaUnaTecla ("Fichero ledo"); }
  else
    { Usr_Avisa ("El fichero no se ha podido leer"); }

  LimpiaZonaTrabajo();
  Cad_Destruye (Completo);
  }

/*--------------------------------------------------------------------
 * FUNCION:  ExaminaTiff()
 * OBJETIVO: Comprobar si el programa puede manejar el Tiff e informar
 *           al usuario de sus caractersticas
 * ENTRADAS: El Tiff
 * SALIDAS:  Lgica
 *------------------------------------------------------------------*/
logico ExaminaTiff (Tiff)
tiff Tiff;
  {
  entero   Ancho, Alto;
  contador Compresion, BitsPorPixel, Interpretacion;
  logico   Vale=SI;
  octeto   Fila;

  Ancho = Tif_Ancho (Tiff);
  Alto  = Tif_Alto (Tiff);
  Pan_Tinta (VERDE);
  Pan_PonTexto (7,3,"Dimensiones: ");
  Pan_Tinta (BLANCO);
  Pan_Entero (Ancho,1);
  Pan_Texto (" x ");
  Pan_Entero (Alto,1);

  Compresion = Tif_Compresion (Tiff);
  Pan_Tinta (VERDE);
  Pan_PonTexto (9,3,"Modo de compresion: ");
  Pan_Tinta (BLANCO);
  switch ( Compresion )
    {
    case TIF_SIN_COMPRESION: Pan_Texto ("(Ninguno)");     break;
    case TIF_CCITT1D:        Pan_Texto ("CCITT 1D");      break;
    case TIF_FAX3:           Pan_Texto ("FAX 3");         break;
    case TIF_FAX4:           Pan_Texto ("FAX 4");         break;
    case TIF_LZW:            Pan_Texto ("LZW");           break;
    case TIF_JPEG:           Pan_Texto ("JPEG");          break;
    case TIF_PACKBITS:       Pan_Texto ("PackBits");      break;
    default:                 Pan_Texto ("(Desconocido)"); break;
    }

  BitsPorPixel = Tif_BitsPorPixel (Tiff);
  Pan_Tinta (VERDE);
  Pan_PonTexto (11,3,"Nmero de colores: ");
  Pan_Tinta (BLANCO);
  Pan_Entero (1<<BitsPorPixel,1);

  Interpretacion = Tif_Interpretacion(Tiff);
  Pan_Tinta (VERDE);
  Pan_PonTexto (13,3,"Interpretacin fotomtrica: ");
  Pan_Tinta (BLANCO);
  switch ( Interpretacion )
    {
    case TIF_BLANCO_CERO: Pan_Texto ("Cero es blanco");    break;
    case TIF_NEGRO_CERO:  Pan_Texto ("Cero es negro");     break;
    case TIF_RGB:         Pan_Texto ("Color real");        break;
    case TIF_PALETA:      Pan_Texto ("Paleta de colores"); break;
    case TIF_MASCARA:     Pan_Texto ("Mscara");           break;
    default:              Pan_Texto ("(Desconocida)");     break;
    }

  Pan_Tinta (VERDE);
  Pan_PonTexto (15,3,"Tamao: ");
  Pan_Tinta (BLANCO);
  Pan_Entero (Tif_Tamano(Tiff),1);
  Pan_Texto (" octetos");

  Fila = 15;
  if ( Ancho > 640 )
    {
    Pan_PonTexto (Fila, 3, "Demasiada anchura");
    Vale = NO;
    Fila++;
    }

  if ( Alto > 480 )
    {
    Pan_PonTexto (Fila, 3, "Demasiada altura");
    Vale = NO;
    Fila++;
    }

  if ( Compresion != TIF_SIN_COMPRESION )
    {
    Pan_PonTexto (Fila, 3, "No se admite el modo de compresion");
    Vale = NO;
    Fila++;
    }

  if ( BitsPorPixel != 4 )
    {
    Pan_PonTexto (Fila, 3, "No se admite el nmero de colores");
    Vale = NO;
    Fila++;
    }

  if ( Interpretacion != TIF_BLANCO_CERO && Interpretacion != TIF_NEGRO_CERO )
    {
    Pan_PonTexto (Fila, 3, "No se admite la interpretacin fotomtrica");
    Vale = NO;
    Fila++;
    }

  if ( !Vale )  { Usr_Avisa ("No puedo manejar esta imagen"); }

  return ( Vale );
  }

/*--------------------------------------------------------------------
 * FUNCION:  LeeBandas()
 * OBJETIVO: Leer desde un tiff sobre las bandas de la imagen
 * ENTRADAS: El tiff
 * SALIDAS:  Lgica
 *------------------------------------------------------------------*/
logico LeeBandas (Tiff)
tiff Tiff;
  {
  dibugris BandaOrigen;
  logico   TodoBien=NO;
  entero   FilaBandaOrigen=0, FilaBandaDestino=0, i;
  memoria  PosOrigen, PosDestino;
  contador NumBandaDestino=0, NumBandaOrigen=0;

  if ( BandaOrigen = Dgr_Crea (Tif_Ancho(Tiff),
                               (entero)Tif_FilasPorBanda(Tiff)) )
    {
    TodoBien = Tif_LeeBandaBN (Tiff, NumBandaOrigen, Dgr_Dato(BandaOrigen));
    PosOrigen = Dgr_Dato(BandaOrigen);
    PosDestino = Dgr_Dato(Banda[NumBandaDestino]);
    for ( i = 0 ; i < AltoTotal && TodoBien ; i++ )
      {
      Mem_Copia (PosDestino, PosOrigen, Dgr_AnchoEnOxel(BandaOrigen));

      FilaBandaOrigen++;
      if ( FilaBandaOrigen == Tif_FilasPorBanda(Tiff) )
        {
        FilaBandaOrigen = 0;
        NumBandaOrigen++;
        if ( NumBandaOrigen < Tif_TotalBanda(Tiff) )
          {
          TodoBien = Tif_LeeBandaBN (Tiff, NumBandaOrigen,
                                     Dgr_Dato(BandaOrigen));
          }
        PosOrigen = Dgr_Dato(BandaOrigen);
        }
      else  { PosOrigen += Dgr_AnchoEnOxel(BandaOrigen); }

      FilaBandaDestino++;
      if ( FilaBandaDestino == Dgr_Alto(Banda[NumBandaDestino]) )
        {
        FilaBandaDestino = 0;
        NumBandaDestino++;
        PosDestino = Dgr_Dato(Banda[NumBandaDestino]);
        }
      else  { PosDestino += Dgr_AnchoEnOxel(BandaOrigen); }
      }
    Dgr_Destruye ( BandaOrigen );
    }

  return ( TodoBien );
  }

/*--------------------------------------------------------------------
 * FUNCION:  GrabaFichero()
 * OBJETIVO: Grabar un fichero TIFF
 * ENTRADAS: Directorio, nombre y extensin del fichero
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void GrabaFichero (Dir,Nom,Ext)
cadena Dir,Nom,Ext;
  {
  tiff   Tiff;
  cadena Completo;
  logico TodoBien=NO;

  Completo = NombreCompleto(Dir,Nom,Ext);
  if ( Tiff = Tif_Crea (Completo) )
    {
    Usr_Informa ("Grabando fichero");
    TodoBien = GrabaBandas(Tiff);
    Tif_Cierra (Tiff);
    }

  if ( TodoBien )
    { Usr_PulsaUnaTecla ("Fichero grabado"); }
  else
    { Usr_Avisa ("El fichero no se ha podido grabar"); }

  Cad_Destruye (Completo);
  }

/*--------------------------------------------------------------------
 * FUNCION:  GrabaBandas()
 * OBJETIVO: Grabar en un tiff las bandas de la imagen
 * ENTRADAS: El tiff
 * SALIDAS:  Lgica
 *------------------------------------------------------------------*/
logico GrabaBandas (Tiff)
tiff Tiff;
  {
  logico   TodoBien;
  contador NumBanda, TotalBanda;

  for ( TotalBanda = 0 ; Banda[TotalBanda] ; TotalBanda++ )
    ;

  TodoBien = Tif_EscribeDirectorioGris (Tiff, Dgr_Ancho(Banda[0]),
                   (entero)AltoTotal, TotalBanda, TIF_SIN_COMPRESION,
                   TIF_NEGRO_CERO);

  for ( NumBanda = 0 ; Banda[NumBanda] && TodoBien ; NumBanda++ )
    {
    TodoBien = Tif_EscribeBandaBN (Tiff, Dgr_Tamano(Banda[NumBanda]),
                     NumBanda, Dgr_Dato(Banda[NumBanda]));
    }

  return ( TodoBien );
  }

/*--------------------------------------------------------------------
 * FUNCION:  NombreCompleto()
 * OBJETIVO: Devolver el nombre completo de un fichero
 * ENTRADAS: Directorio, nombre y extensin del fichero
 * SALIDAS:  Una cadena con el nombre completo
 * NOTA:     La cadena hay que destruirla cuando no sea necesaria
 *------------------------------------------------------------------*/
cadena NombreCompleto (Dir,Nom,Ext)
cadena Dir,Nom,Ext;
  {
  cadena Respuesta, Aux;

  Aux = Fch_Nombre (Nom,Ext);
  Respuesta = Cad_Une (Dir,Aux,CAD_FIN);
  Cad_Destruye (Aux);

  return ( Respuesta );
  }

/*--------------------------------------------------------------------
 * FUNCION:  EligeNombre()
 * OBJETIVO: Elegir un nombre de fichero de entre los posibles
 * ENTRADAS: Directorio, mscara y extensin del fichero
 * SALIDAS:  Un nombre, sin directorio ni extensin
 * NOTA:     El nombre hay que destruirlo cuando no sea necesario
 *------------------------------------------------------------------*/
cadena EligeNombre (Dir,Mas,Ext)
cadena Dir,Mas,Ext;
  {
  cadena   Respuesta, Completo, Mensaje;
  lista    Posible;
  entero   Elegido;
  region   Region;
  menu     Eleccion;
  caracter Aux[81];
  octeto   i;
  logico   Visto = NO;

  Region = Reg_Crea (11,30,21,50);
  Reg_Borde (Region, CDR_SIMPLE, NEGRO, VERDE);
  Completo = NombreCompleto(Dir,Mas,Ext);
  Usr_Informa ("Buscando ficheros");
  if ( Posible = Fch_ListaNombre (Completo) )
    {
    if ( HayQueOrdenar )
      { Posible = Lis_Ordena (Posible, ComparaCadena); }
    Eleccion = Men_Crea (MEN_VERTICAL,
                         Reg_FilSup(Region)+1, Reg_ColIzq(Region)+1,
                         Reg_FilInf(Region)-1, Reg_ColDer(Region)-1,
                         Lis_Total(Posible), MEN_LISTA, (memoria)Posible);
    if ( Elegido = Men_Ejecuta (Eleccion,(entero)1,
                   "Elige uno de los ficheros") )
      {
      Men_Texto (Eleccion, Elegido, Aux);
      for ( i=Cad_Longitud(Aux)-1 ; i && !Visto ; i-- )
        {
        if ( Aux[i] == Fch_Separador() )
          {
          Aux[i] = NULO;
          Visto = SI;
          }
        }
      Respuesta = Cad_Duplica (Aux);
      }
    else  { Respuesta = Cad_Duplica (""); }

    Lis_Destruye (Posible);
    Men_Destruye (Eleccion);
    }

  else
    {
    Mensaje = Cad_Une ("No se encontrado ningn fichero ","\"",Completo,
                       "\"", CAD_FIN);
    Usr_PulsaUnaTecla (Mensaje);
    Cad_Destruye (Mensaje);
    Respuesta = Cad_Duplica ("");
    }

  Reg_Destruye (Region);
  Cad_Destruye (Completo);

  return ( Respuesta );
  }

/*--------------------------------------------------------------------
 * FUNCION:  ComparaCadena()
 * OBJETIVO: Comparar dos cadenas
 * ENTRADAS: Los punteros a las dos cadenas
 * SALIDAS:  Un int que indica qu cadena es menor
 *------------------------------------------------------------------*/
int ComparaCadena (Cad1,Cad2)
cadena *Cad1, *Cad2;
  {
  return ( Cad_Compara (*Cad1,*Cad2) );
  }

/*--------------------------------------------------------------------
 * FUNCION:  PideNombre()
 * OBJETIVO: Pedir al usuario un nombre
 * ENTRADAS: El nombre propuesto
 * SALIDAS:  El nuevo nombre
 *------------------------------------------------------------------*/
cadena PideNombre (Nombre)
cadena Nombre;
  {
  region Region;
  cadena Respuesta;

  Region = Reg_Crea (11,27,17,54);
  Reg_Borde (Region,CDR_SIMPLE,NEGRO,VERDE);
  Pan_Color (NEGRO,VERDE);
  Reg_TextoCentrado (Region,"Escribe el nombre:",2);
  Respuesta = Usr_Texto (Nombre, LONG_NOMBRE,
                         Reg_FilSup(Region)+4,Reg_ColIzq(Region)+7,
                         ROJO, BLANCO);
  Reg_Destruye (Region);
  return ( Respuesta );
  }

/*--------------------------------------------------------------------
 * FUNCION:  PideDirectorio()
 * OBJETIVO: Pedir al usuario un directorio
 * ENTRADAS: El directorio propuesto
 * SALIDAS:  El nuevo directorio
 *------------------------------------------------------------------*/
cadena PideDirectorio (Dir)
cadena Dir;
  {
  region Region;
  cadena Respuesta;

  Region = Reg_Crea (11,22,17,59);
  Reg_Borde (Region,CDR_SIMPLE,NEGRO,VERDE);
  Pan_Color (NEGRO,VERDE);
  Reg_TextoCentrado (Region,"Escribe el directorio:",2);
  Respuesta = Usr_Texto (Dir, LONG_DIRECTORIO,
                         Reg_FilSup(Region)+4,Reg_ColIzq(Region)+7,
                         ROJO, BLANCO);
  Reg_Destruye (Region);
  return ( Respuesta );
  }

/*--------------------------------------------------------------------
 * FUNCION:  PideMascara()
 * OBJETIVO: Pedir al usuario una mscara
 * ENTRADAS: La mscara propuesta
 * SALIDAS:  La nueva mscara
 *------------------------------------------------------------------*/
cadena PideMascara (Mascara)
cadena Mascara;
  {
  region Region;
  cadena Respuesta;

  Region = Reg_Crea (11,22,17,59);
  Reg_Borde (Region,CDR_SIMPLE,NEGRO,VERDE);
  Pan_Color (NEGRO,VERDE);
  Reg_TextoCentrado (Region,"Escribe la mscara:",2);
  Respuesta = Usr_Texto (Mascara, LONG_MASCARA,
                         Reg_FilSup(Region)+4,Reg_ColIzq(Region)+14,
                         ROJO, BLANCO);
  Reg_Destruye (Region);
  return ( Respuesta );
  }

/*--------------------------------------------------------------------
 * FUNCION:  PideExtension()
 * OBJETIVO: Pedir al usuario una extensin
 * ENTRADAS: La extensin propuesta
 * SALIDAS:  La nueva extensin
 *------------------------------------------------------------------*/
cadena PideExtension (Extension)
cadena Extension;
  {
  region Region;
  cadena Respuesta;

  Region = Reg_Crea (11,22,17,59);
  Reg_Borde (Region,CDR_SIMPLE,NEGRO,VERDE);
  Pan_Color (NEGRO,VERDE);
  Reg_TextoCentrado (Region,"Escribe la extensin:",2);
  Respuesta = Usr_Texto (Extension, LONG_EXTENSION,
                         Reg_FilSup(Region)+4,Reg_ColIzq(Region)+16,
                         ROJO, BLANCO);
  Reg_Destruye (Region);
  return ( Respuesta );
  }

/*--------------------------------------------------------------------
 * FUNCION:  ManejoImagen()
 * OBJETIVO: Manejar las opciones de creacin de imgenes
 * ENTRADAS: Ninguna
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void ManejoImagen()
  {
  enum { SALIDA_ESC, FORMULA, GENERAR, MOSTRAR, TAMANO, ZONA, INVERTIR,
         BORRAR };
  static cadena MenuImagen[] = {">Frmula", ">Generar", ">Mostrar",
                                ">Tamao", ">Zona", ">Invertir", ">Borrar",
                                NIL };
  static caracter Formula[LONG_FORMULA+1] = "";
  static memoria  Programa = NIL;

  region  Region;
  trozo   MenuAnt;
  entero  Opcion = FORMULA;
  logico  Sigue = SI, Refresca = SI, Generado = NO;
  cadena  Aux;

  if ( Region = Reg_Crea (4,20,12,61) )
    {
    Reg_Color (Region,NEGRO,VERDE);
    while ( Sigue )
      {
      if ( Refresca )
        {
        Reg_Borde (Region,CDR_SIMPLE,NEGRO,VERDE);
        Pan_Borra (NEGRO,Reg_FilSup(Region)+1, Reg_ColIzq(Region)+13,
                         Reg_FilInf(Region)-1, Reg_ColDer(Region)-1);
        Reg_PonTexto (Region,1,13,Formula);
        if ( Generado )  { Reg_PonTexto (Region,2,13,"(Hecho)"); }
        Reg_PonEntero (Region,4,13,Dgr_Ancho(Banda[0]),1);
        Pan_Texto (" x ");
        Pan_Entero (AltoTotal,1);
        Reg_PonCar (Region,5,13,'(');
        Pan_Entero (FilSup,1);
        Pan_Caracter (',');
        Pan_Entero (ColIzq,1);
        Pan_Texto (") - (");
        Pan_Entero (FilInf,1);
        Pan_Caracter (',');
        Pan_Entero (ColDer,1);
        Pan_Caracter (')');
        Refresca = NO;
        }

      Opcion = Men_Vertical (Reg_FilSup(Region)+1, Reg_ColIzq(Region)+1,
                             Reg_FilInf(Region)-1, Reg_ColIzq(Region)+12,
                             MenuImagen, Opcion);
      switch ( Opcion )
        {
        case SALIDA_ESC:  Sigue = NO;      break;

        case FORMULA:     Aux = PideFormula (Formula);
                          if ( Tec_Ultima() != TEC_ESC )
                            {
                            Cad_Copia (Formula, Aux);
                            if ( Programa )  { Mem_Destruye (Programa); }
                            Programa = ConvierteCadenaEnPrograma (Formula);
                            Refresca = SI;
                            Generado = NO;
                            Opcion = GENERAR;
                            }
                          Cad_Destruye (Aux);
                          break;

        case GENERAR:     if ( Programa )
                            {
                            GeneraImagen (Programa);
                            Generado = SI;
                            Refresca = SI;
                            Opcion = MOSTRAR;
                            }
                          else
                            { Usr_Avisa ("Primero define la frmula"); }
                          break;

        case MOSTRAR:     MenuAnt = Trz_Crea (0,0,3,79);
                          MuestraImagen();
                          Refresca = SI;
                          if ( MenuAnt )
                            {
                            Trz_Restituye (MenuAnt, 0, 0);
                            Trz_Destruye (MenuAnt);
                            }
                          break;

        case TAMANO:      PideTamano();
                          if ( Tec_Ultima() != TEC_ESC )
                            {
                            Refresca = SI;
                            Generado = NO;
                            Opcion = GENERAR;
                            }
                          break;

        case ZONA:        PideZona();
                          if ( Tec_Ultima() != TEC_ESC )
                            {
                            Refresca = SI;
                            Generado = NO;
                            Opcion = GENERAR;
                            }
                          break;

        case INVERTIR:    Usr_Informa ("Invirtiendo la imagen");
                          InvierteBandas();
                          Generado = NO;
                          Opcion = MOSTRAR;
                          break;

        case BORRAR:      if ( Usr_Consulta ("Quieres borrar la imagen?") )
                            {
                            Usr_Informa ("Borrando la imagen");
                            BorraBandas();
                            Generado = NO;
                            Refresca = SI;
                            Opcion = FORMULA;
                            }
                          break;
        }
      LimpiaZonaImagen();
      }
    Reg_Destruye (Region);
    }
  else  { Usr_Avisa ("Falta memoria"); }
  }

/*--------------------------------------------------------------------
 * FUNCION:  PideFormula()
 * OBJETIVO: Pedir al usuario la frmula para calcular imgenes
 * ENTRADAS: La frmula anterior
 * SALIDAS:  La nueva frmula
 *------------------------------------------------------------------*/
cadena PideFormula (Formula)
cadena Formula;
  {
  region Region;
  cadena Nueva;

  Region = Reg_Crea (14,22,20,59);
  Reg_Borde (Region,CDR_SIMPLE,NEGRO,VERDE);
  Pan_Color (NEGRO,VERDE);
  Reg_TextoCentrado (Region,"Escribe la frmula:",2);
  Nueva = Usr_Texto (Formula, LONG_FORMULA,
                     Reg_FilSup(Region)+4,Reg_ColIzq(Region)+2,
                     ROJO, BLANCO);
  Reg_Destruye (Region);
  return ( Nueva );
  }

/*--------------------------------------------------------------------
 * FUNCION:  ConvierteCadenaEnPrograma()
 * OBJETIVO: Convertir una cadena en un programa para una
 *           calculadora
 * ENTRADAS: Una cadena, que admite estos tokens:
 *           + - * / ALE INT DUP CDR CUB nmeros
 * SALIDAS:  El programa, en una porcin de memoria
 * NOTA:     Cuando no sea necesario, hay que destruir el
 *           programa con Mem_Destruye()
 * ALGORITMO:
 *      Hacer una copia de la cadena y ponerla en maysculas
 *      Cambiar los tabuladores y fines de lnea por espacios
 *      Calcular cuntos espacios hay
 *      Reservar memoria para el programa, teniendo en cuenta
 *        que cada tokens se convierte en tres octetos o menos
 *      Estudiar cada token de la cadena y formar el programa
 *------------------------------------------------------------------*/
memoria ConvierteCadenaEnPrograma (Cadena)
cadena Cadena;
  {
  memoria Programa;
  cadena  Trabajo, Token;
  entero  i, Long, Numero;

  Trabajo = Cad_Mayus (Cad_Duplica(Cadena));
  Cad_Cambia (Trabajo, '\t', ' ');
  Cad_Cambia (Trabajo, '\r', ' ');
  Cad_Cambia (Trabajo, '\n', ' ');

  for ( i=0 , Long=0 ; Trabajo[i] ; i++ )
    { if ( Trabajo[i] == ' ' )  { Long++; } }

  if ( Programa = Mem_Crea ( 3*(Long+1)+1 ) )
    {
    Token = strtok (Trabajo, " ");
    i = 0;
    while ( Token )
      {
      if ( Cad_Igual (Token, "+") || Cad_Igual (Token, "-") ||
           Cad_Igual (Token, "*") || Cad_Igual (Token, "/") ||
           Cad_Igual (Token, "ALE") )
        {
        Programa[i++] = CAL_OPERACION;
        switch ( Token[0] )
          {
          case '+': Programa[i++] = CAL_SUMA;       break;
          case '-': Programa[i++] = CAL_DIFERENCIA; break;
          case '*': Programa[i++] = CAL_PRODUCTO;   break;
          case '/': Programa[i++] = CAL_COCIENTE;   break;
          case 'A': Programa[i++] = CAL_ALEATORIO;  break;
          }
        }

      else if ( Cad_Igual (Token, "CDR") || Cad_Igual (Token, "CUB") )
        {
        Programa[i++] = CAL_FUNCION;
        switch ( Token[1] )
          {
          case 'D': Programa[i++] = CAL_CUADRADO; break;
          case 'U': Programa[i++] = CAL_CUBO;     break;
          }
        }

      else if ( Cad_Igual (Token, "INT") )
        { Programa[i++] = CAL_INTERCAMBIA; }

      else if ( Cad_Igual (Token, "DUP") )
        { Programa[i++] = CAL_DUPLICA; }

      else  /* El token debe ser un nmero */
        {
        Numero = atoi (Token);
        if ( Numero >= 0 )
          { Programa[i++] = CAL_NUMERO_POS; }
        else
          {
          Programa[i++] = CAL_NUMERO_NEG;
          Numero *= -1;
          }
        Programa[i++] = Numero / 256;
        Programa[i++] = Numero % 256;
        }

      Token = strtok (NULL, " ");
      }  /* Fin while Token */
    Programa[i] = CAL_FIN_PROGRAMA;
    }  /* Fin if Programa */

  Cad_Destruye (Trabajo);
  return ( Programa );
  }

/*--------------------------------------------------------------------
 * FUNCION:  MuestraImagen()
 * OBJETIVO: Mostrar la imagen que haya en memoria
 * ENTRADAS: Ninguna, se utiliza la variable global Bandas
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void MuestraImagen()
  {
  entero i, j, NumBanda=0, FilaEnBanda=0;

  Pan_Borra (NEGRO,0,0,23,79);
  PreparaColor();

  for ( i=0 ; i<AltoTotal ; i++ )
    {
    for ( j=0 ; j<Dgr_Ancho(Banda[0]) ; j++ )
      {
      Zon_PixelFisico ((contador)j, (contador)i,
                       Dgr_Punto(Banda[NumBanda], FilaEnBanda, j));
      }
    FilaEnBanda++;
    if ( FilaEnBanda == Dgr_Alto(Banda[NumBanda]) )
      {
      FilaEnBanda = 0;
      NumBanda++;
      }
    }

  Son_Bien();
  Tec_Pulsada();
  BorraPantallaVGA();
  RestauraColor();
  }

/*--------------------------------------------------------------------
 * FUNCION:  GeneraImagen()
 * OBJETIVO: Generar una imagen a partir de una frmula
 * ENTRADAS: El programa que define la frmula
 * SALIDAS:  Cambia la variable global Bandas[]
 *------------------------------------------------------------------*/
void GeneraImagen (Programa)
memoria Programa;
  {
  entero   i, j, NumBanda=0, FilaEnBanda=0;
  region   Region;
  calculad Calculadora;

  Usr_Informa ("Generando imagen");

  Calculadora = Cal_Crea (5);

  Region = Reg_Crea (14,31,18,48);
  Reg_Borde (Region,CDR_SIMPLE,NEGRO,VERDE);
  Reg_Color (Region,NEGRO,VERDE);
  Reg_PonTexto (Region,2,4,"Fila:");
  Reg_Color (Region,NEGRO,BLANCO);

  for ( i=0 ; i<FilSup ; i++ )
    {
    FilaEnBanda++;
    if ( FilaEnBanda == Dgr_Alto(Banda[NumBanda]) )
      {
      FilaEnBanda = 0;
      NumBanda++;
      }
    }

  for ( i=FilSup ; i<=FilInf ; i++ )
    {
    Reg_PonEntero (Region,2,10,i,1);
    for ( j=ColIzq ; j<=ColDer ; j++ )
      {
      Cal_Inserta (Calculadora, i);
      Cal_Inserta (Calculadora, j);
      Dgr_PonPunto(Banda[NumBanda], FilaEnBanda, j,
                   (octeto)(Cal_Ejecuta(Calculadora,Programa)%16));
      }
    FilaEnBanda++;
    if ( FilaEnBanda == Dgr_Alto(Banda[NumBanda]) )
      {
      FilaEnBanda = 0;
      NumBanda++;
      }
    }

  Reg_Destruye (Region);
  Cal_Destruye (Calculadora);
  }

/*--------------------------------------------------------------------
 * FUNCION:  PideTamano()
 * OBJETIVO: Pedir al usuario las dimensiones de la imagen y
 *           cambiarla en consecuencia
 * ENTRADAS: Ninguna
 * SALIDAS:  Ninguna, la variable Bandas[] puede cambiar
 *------------------------------------------------------------------*/
void PideTamano()
  {
  region  Region;
  contpos Ancho, Alto;

  Region = Reg_Crea (13,25,21,56);
  Reg_Borde (Region,CDR_SIMPLE,NEGRO,VERDE);
  Pan_Color (NEGRO,VERDE);
  Reg_TextoCentrado (Region,"Escribe el nuevo tamao:",2);

  Reg_PonTexto (Region,4,8,"Anchura:");
  Ancho = Usr_Entero ((contador)Dgr_Ancho(Banda[0]), 3, 1, 640,
                      Reg_FilSup(Region)+4,Reg_ColIzq(Region)+20,
                      ROJO, BLANCO);
  if ( Tec_Ultima() != TEC_ESC )
    {
    Reg_PonTexto (Region,6,8,"Altura:");
    Alto = Usr_Entero ((contador)AltoTotal, 3, 1, 480,
                       Reg_FilSup(Region)+6,Reg_ColIzq(Region)+20,
                       ROJO, BLANCO);
    if ( Tec_Ultima() != TEC_ESC )
      { CreaBandas (Ancho, Alto); }
    }
  Reg_Destruye (Region);
  }

/*--------------------------------------------------------------------
 * FUNCION:  PideZona()
 * OBJETIVO: Pedir al usuario las coordenadas de la zona y
 *           cambiarla en consecuencia
 * ENTRADAS: Ninguna
 * SALIDAS:  Ninguna, las variables de la zona pueden cambiar
 *------------------------------------------------------------------*/
void PideZona()
  {
  region   Region;
  contador Sup, Inf, Izq, Der;

  Region = Reg_Crea (13,25,21,56);
  Reg_Borde (Region,CDR_SIMPLE,NEGRO,VERDE);
  Pan_Color (NEGRO,VERDE);
  Reg_PonTexto (Region,2,4,"Escribe la nueva zona:");

  Reg_PonTexto (Region,4,4,"Filas, de");
  Sup = Usr_Entero (FilSup, 3, 0, AltoTotal-1,
                    Reg_FilSup(Region)+4,Reg_ColIzq(Region)+14,
                    ROJO, BLANCO);
  if ( Tec_Ultima() != TEC_ESC )
    {
    Reg_PonCar (Region,4,18,'a');
    Inf = Usr_Entero (FilInf, 3, Sup, AltoTotal-1,
                      Reg_FilSup(Region)+4,Reg_ColIzq(Region)+20,
                      ROJO, BLANCO);
    if ( Tec_Ultima() != TEC_ESC )
      {
      Reg_PonTexto (Region,6,4,"Columnas, de");
      Izq = Usr_Entero (ColIzq, 3, 0, (contador)Dgr_Ancho(Banda[0])-1,
                        Reg_FilSup(Region)+6,Reg_ColIzq(Region)+17,
                        ROJO, BLANCO);
      if ( Tec_Ultima() != TEC_ESC )
        {
        Reg_PonCar (Region,6,21,'a');
        Der = Usr_Entero (ColDer, 3, Izq, (contador)Dgr_Ancho(Banda[0])-1,
                          Reg_FilSup(Region)+6,Reg_ColIzq(Region)+23,
                          ROJO, BLANCO);
        if ( Tec_Ultima() != TEC_ESC )
          {
          FilSup = Sup;
          FilInf = Inf;
          ColIzq = Izq;
          ColDer = Der;
          }
        }
      }
    }
  Reg_Destruye (Region);
  }

/*--------------------------------------------------------------------
 * FUNCION:  CreaBandas()
 * OBJETIVO: Crear las bandas necesarias para manejar una imagen
 * ENTRADAS: La anchura y altura en pixels de la imagen
 * SALIDAS:  Lgica, que indica si se han podido crear las bandas
 *           La variable global Bandas[] puede cambiar
 * EJEMPLO:  CreaBandas (640, 480)
 *------------------------------------------------------------------*/
logico CreaBandas (Ancho, Alto)
contpos Ancho, Alto;
  {
  logico   TodoBien = SI;
  entero   Tamano, AnchoEnOxel, AltoBanda;
  contador TotalBanda, NumBanda;

  /* Destruimos las bandas existentes */
  for ( NumBanda = 0 ; Banda[NumBanda] ; NumBanda++ )
    {
    Dgr_Destruye (Banda[NumBanda]);
    Banda[NumBanda] = NIL;
    }

  AltoTotal = Alto;
  AnchoEnOxel = (Ancho+1)/2;
  Tamano = AnchoEnOxel * Alto;
  TotalBanda = DivEx (Tamano, MAX_TAMANO_BANDA);
  AltoBanda = DivEx (Alto, TotalBanda);

  for ( NumBanda = 0 ; NumBanda < TotalBanda-1 && TodoBien ; NumBanda++ )
    {
    Banda[NumBanda] = Dgr_Crea ((entero)Ancho, AltoBanda);
    if ( !Banda[NumBanda] )  { TodoBien = NO; }
    }
  if ( TodoBien )
    {
    /* La ltima banda se trata de modo especial
       ya que puede ser ms pequea */
    AltoBanda = Alto - (TotalBanda-1)*AltoBanda;
    Banda[NumBanda] = Dgr_Crea ((entero)Ancho, AltoBanda);
    if ( !Banda[NumBanda] )  { TodoBien = NO; }
    else                     { Banda[++NumBanda] = NIL; }
    }

  FilSup = 0;
  ColIzq = 0;
  if ( TodoBien )
    {
    FilInf = AltoTotal-1;
    ColDer = Dgr_Ancho(Banda[0])-1;
    }
  else
    {
    FilInf = 0;
    ColDer = 0;
    }

  return ( TodoBien );
  }

/*--------------------------------------------------------------------
 * FUNCION:  InvierteBandas()
 * OBJETIVO: Invertir los tonos de todas las bandas de la imagen
 * ENTRADAS: Ninguna
 * SALIDAS:  Ninguna, la variable global Bandas[] cambia
 *------------------------------------------------------------------*/
void InvierteBandas()
  {
  contador NumBanda;

  for ( NumBanda = 0 ; Banda[NumBanda] ; NumBanda++ )
    { Dgr_Invierte (Banda[NumBanda]); }
  }

/*--------------------------------------------------------------------
 * FUNCION:  BorraBandas()
 * OBJETIVO: Borrar todas las bandas de la imagen
 * ENTRADAS: Ninguna
 * SALIDAS:  Ninguna, la variable global Bandas[] cambia
 *------------------------------------------------------------------*/
void BorraBandas()
  {
  contador NumBanda;

  for ( NumBanda = 0 ; Banda[NumBanda] ; NumBanda++ )
    { Dgr_Borra (Banda[NumBanda]); }
  }

/*--------------------------------------------------------------------
 * FUNCION:  Opciones()
 * OBJETIVO: Seleccionar las opciones del programa
 * ENTRADAS: Ninguna
 * SALIDAS:  Las variables globales pueden modificarse
 *------------------------------------------------------------------*/
void Opciones()
  {
  enum { SALIDA_ESC, SONIDO, ORDEN };
  static cadena MenuOpc[] = {">Sonido", ">Orden", NIL };
  entero Opcion = SONIDO;
  logico Sigue = SI;
  region Region;

  if ( Region = Reg_Crea (4,59,7,79) )
    {
    Reg_Borde (Region,CDR_SIMPLE,NEGRO,VERDE);
    while ( Sigue )
      {
      Pan_Color (NEGRO, VERDE);
      Reg_Cursor (Region,1,10);
      if ( Son_Estado() )  { Pan_Texto ("Encendido"); }
      else                 { Pan_Texto ("Apagado  "); }

      Reg_Cursor (Region,2,10);
      if ( HayQueOrdenar )  { Pan_Texto ("S"); }
      else                  { Pan_Texto ("No"); }

      Opcion = Men_Vertical (Reg_FilSup(Region)+1, Reg_ColIzq(Region)+1,
                             Reg_FilInf(Region)-1, Reg_ColIzq(Region)+9,
                             MenuOpc, Opcion);
      switch ( Opcion )
        {
        case SALIDA_ESC: Sigue = NO;                      break;
        case SONIDO:     Son_Cambia();                    break;
        case ORDEN:      HayQueOrdenar = !HayQueOrdenar;  break;
        }
      }
    Reg_Destruye (Region);
    }
  else  { Usr_Avisa ("Falta memoria"); }
  }

/*--------------------------------------------------------------------
 * Funciones de manejo de colores
 *------------------------------------------------------------------*/

/*--------------------------------------------------------------------
 * FUNCION:  PreparaColor()
 * OBJETIVO: Redefinir los colores de la VGA para usar 16 niveles
 *           de gris
 * ENTRADAS: Ninguna
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void PreparaColor()
  {
  octeto k;

  for ( k=0 ; k<16 ; k++ )
    {
    AsignaPaletaDAC (k, k);
    CambiaColorDAC (k, 4*k, 4*k, 4*k);
    }
  }

/*--------------------------------------------------------------------
 * FUNCION:  AsignaPaletaDAC()
 * OBJETIVO: Cambiar el registro DAC asignado a un registro de paleta
 * ENTRADAS: El nmero de registro de paleta (de 0 a 15) y el del
 *           registro DAC (de 0 a 255)
 * SALIDAS:  Ninguna
 * NOTA:     No estoy seguro de la correccin de la documentacin
 *           de esta funcin
 *------------------------------------------------------------------*/
void AsignaPaletaDAC (Paleta, DAC)
octeto Paleta, DAC;
  {
  union REGS r;

  r.h.ah = 0x10;
  r.h.al = 0x0;
  r.h.bh = DAC;
  r.h.bl = Paleta;

  int86 (0x10,&r,&r);
  }

/*--------------------------------------------------------------------
 * FUNCION:  CambiaColorDAC()
 * OBJETIVO: Cambiar el color asociado a un registro DAC
 * ENTRADAS: El nmero de registro (de 0 a 255) y los valores RGB del
 *           color (de 0 a 63)
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void CambiaColorDAC (Registro, Rojo, Verde, Azul)
octeto Registro, Rojo, Verde, Azul;
  {
  union REGS r;

  r.h.ah = 0x10;
  r.h.al = 0x10;
  r.x.bx = Registro;
  r.h.dh = Rojo;
  r.h.ch = Verde;
  r.h.cl = Azul;

  int86 (0x10,&r,&r);
  }

/*--------------------------------------------------------------------
 * FUNCION:  RestauraColor()
 * OBJETIVO: Redefinir los colores usados por Olimpo en modo grfico
 * ENTRADAS: Ninguna
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void RestauraColor()
  {
  CambiaColorDAC (NEGRO,  0x00, 0x00, 0x00);
  CambiaColorDAC (ROJO,   0x2A, 0x00, 0x00);
  CambiaColorDAC (VERDE,  0x00, 0x2A, 0x00);
  CambiaColorDAC (BLANCO, 0x2A, 0x2A, 0x2A);
  }
