/*--------------------------------------------------------------------
 * FICHERO:  Tiff_G16.c
 * OBJETIVO: Definir la funcin Tif_EscribeDirectorioGris()
 * AUTOR:    Pedro Reina
 * FECHA:    D.21.7.1996
 *------------------------------------------------------------------*/

/*--------------------------------------------------------------------
 * Ficheros de cabecera
 *------------------------------------------------------------------*/

#include "Tiff_G16.h"

/*--------------------------------------------------------------------
 * Definicin de macros constantes
 *------------------------------------------------------------------*/

#define TIF_DIRECTORIO_GRIS    126

/*--------------------------------------------------------------------
 * Definicin de funciones
 *------------------------------------------------------------------*/

/*--------------------------------------------------------------------
 * FUNCION:  Tif_EscribeDirectorioGris()
 * OBJETIVO: Escribir el directorio de una imagen en 16 niveles
 *           de gris en un fichero TIFF
 * ENTRADAS: Un objeto tiff, la anchura y altura de la imagen,
 *           el nmero de bandas en que se va a dividir la
 *           imagen, el mtodo de compresin y la interpretacin
 *           fotomtrica
 * SALIDAS:  Lgica, que indica que todo ha ido bien
 * EJEMPLO:  Tif_EscribeDirectorioGris (Imagen, 240, 100,
 *                     1, TIF_SIN_COMPRESION, TIF_NEGRO_CERO)
 * NOTAS:    1. El modo de compresin slo puede ser
 *              TIF_SIN_COMPRESION
 *           2. La interpretacin fotomtrica puede ser
 *              TIF_NEGRO_CERO o TIF_BLANCO_CERO
 *           3. Se asume que las bandas sern del mismo tamao,
 *              salvo quiz la ltima
 * ALGORITMO:
 *      El orden en que se van a grabar los datos es:
 *         El directorio
 *         Los punteros a las bandas
 *         Los punteros a los tamaos a las bandas
 *         Los datos de cada banda
 *
 *      Colocarse al final del fichero
 *      Si la posicin es impar, aadir un octeto
 *      Anotar en el fichero la posicin del nuevo directorio
 *      Anotar en memoria la posicin del nuevo directorio
 *      Calcular los valores necesarios y anotarlos en el directorio
 *      Escribir el directorio en el fichero y, si es necesario,
 *        la informacin adicional
 *------------------------------------------------------------------*/
logico Tif_EscribeDirectorioGris (Tiff, Ancho, Alto, TotalBanda,
                                  Compresion, Interpretacion)
tiff     Tiff;
entero   Ancho, Alto;
contador TotalBanda;
contpos  Compresion, Interpretacion;
  {
  static octeto Directorio[TIF_DIRECTORIO_GRIS] = {
    10, 0,           /*  Hay 10 tags en este directorio         */
    254,0, 4,0, 1,0,0,0, 0,0,0,0,    /*  NewSubfileType         */
    0,1,   3,0, 1,0,0,0, 0,0,0,0,    /*  ImageWith              */
    1,1,   3,0, 1,0,0,0, 0,0,0,0,    /*  ImageLength            */
    2,1,   3,0, 1,0,0,0, 4,0,0,0,    /*  BitsPerSample          */
    3,1,   3,0, 1,0,0,0, 0,0,0,0,    /*  Compression            */
    6,1,   3,0, 1,0,0,0, 0,0,0,0,    /*  PhotometricInterpret   */
    17,1,  4,0, 1,0,0,0, 0,0,0,0,    /*  StripOffsets           */
    21,1,  3,0, 1,0,0,0, 1,0,0,0,    /*  SamplesPerPixel        */
    22,1,  4,0, 1,0,0,0, 0,0,0,0,    /*  RowsPerStrip           */
    23,1,  4,0, 1,0,0,0, 0,0,0,0,    /*  StripByteCounts        */
    0, 0, 0, 0    /*  Puntero al siguiente directorio de imagen */
    };

  octeto   Aux[4] = {0, 0, 0, 0};
  logico   Error = NO;
  entero   Pos;
  contador i;

  if ( Fch_ColocaFinal (Tif_Fichero(Tiff)) == -1 )
    { Error = SI; }
  else
    { Pos = Fch_Posicion (Tif_Fichero(Tiff)); }

  if ( !Error )
    {
    if ( Pos % 2 )
      {
      Error = !Fch_EscribeOcteto (Tif_Fichero(Tiff), Aux, 1);
      Pos++;
      }
    }

  if ( !Error )
    {
    Mem_EscribeEntero (Aux, MEM_INTEL, Pos);
    Fch_Coloca (Tif_Fichero(Tiff), Tif_ComienzoDir(Tiff) + 2 +
                                   TIF_LONGITUDTAG * Tif_TotalTag(Tiff));
    Error = !Fch_EscribeOcteto (Tif_Fichero(Tiff), Aux, 4);
    Tif_PonComienzoDir (Tiff, Pos);
    }

  if ( !Error )
    {
    Tif_PonTotalTag (Tiff, 10);
    Tif_PonMetodoCompr (Tiff, Compresion);
    Tif_PonTotalBanda (Tiff, TotalBanda);
    Mem_EscribeEntero (Directorio+22, MEM_INTEL, Ancho);
    Mem_EscribeEntero (Directorio+34, MEM_INTEL, Alto);
    Mem_EscribeEntero (Directorio+58, MEM_INTEL, Compresion);
    Mem_EscribeEntero (Directorio+70, MEM_INTEL, Interpretacion);
    Mem_EscribeEntero (Directorio+82, MEM_INTEL, Pos+TIF_DIRECTORIO_GRIS);

    if ( TotalBanda == 1 )
      {
      Mem_EscribeEntero (Directorio+106, MEM_INTEL, Alto);
      Mem_EscribeEntero (Directorio+118, MEM_INTEL, Alto*((Ancho+1)/2));
      }

    else
      {
      Mem_EscribeEntero (Directorio+78, MEM_INTEL, (entero)TotalBanda);
      Mem_EscribeEntero (Directorio+106, MEM_INTEL, DivEx(Alto,TotalBanda));
      Mem_EscribeEntero (Directorio+114, MEM_INTEL, (entero)TotalBanda);
      Mem_EscribeEntero (Directorio+118, MEM_INTEL,
                                         Pos+TIF_DIRECTORIO_GRIS+4*TotalBanda);
      }
    }

  if ( !Error )
    {
    Fch_ColocaFinal (Tif_Fichero(Tiff));
    Error = !Fch_EscribeOcteto (Tif_Fichero(Tiff), Directorio,
                                TIF_DIRECTORIO_GRIS);
    if ( TotalBanda > 1 )
      {
      for ( i=0 ; i<4 ; i++ )  { Aux[i] = 0; }
      for ( i=0 ; i<2*TotalBanda && !Error ; i++ )
        { Error = !Fch_EscribeOcteto (Tif_Fichero(Tiff), Aux, 4); }
      }
    }

  return ( !Error );
  }
