/*--------------------------------------------------------------------
 * FICHERO:  QuillRTF.c
 * OBJETIVO: Convertir un fichero Quill a Rich Text Format
 * AUTOR:    Pedro Reina
 * FECHA:    X.4.3.1998
 *------------------------------------------------------------------*/

/*--------------------------------------------------------------------
 * Ficheros de cabecera
 *------------------------------------------------------------------*/

#include <Olimpo.h>            /*  Versin 2.0  */
#include <string.h>            /*  strtok()     */
#include "Quill.h"

/*--------------------------------------------------------------------
 * Declaracin de funciones
 *------------------------------------------------------------------*/

void LeeConfig();

void Explica();
void ExplicaObjetivo();
void ExplicaPrograma();
void ExplicaAutor();

void   Fichero();
cadena EligeUnNombre();
cadena PideDirectorio();
cadena PideExtension();
cadena PideMascara();
cadena PideNombre();
cadena NombreCompleto();
void   TrataFichero();
logico CompruebaQuill();
void   Convierte();
void   ExplicaEstado();

void   Opciones();
octeto PideTamano();
octeto PidePosicion();

void Rtf_EscribeCadena();
void Rtf_NuevaLinea();
void Rtf_EscribeCaracter();

/*--------------------------------------------------------------------
 * Definicin de macros constantes
 *------------------------------------------------------------------*/

#define LONG_DIRECTORIO  24
#define LONG_EXTENSION    3
#define LONG_NOMBRE      12
#define LONG_MASCARA     12
#define LONG_DESTINO     20

#define MAX_FICHERO     200

/*--------------------------------------------------------------------
 * Definicin de macros funciones
 *------------------------------------------------------------------*/

#define LimpiaZonaTrabajo()     Pan_Borra(NEGRO,4,0,21,79)
#define LimpiaZonaPequena()     Pan_Borra(NEGRO,10,0,21,79)

/*--------------------------------------------------------------------
 * MACROS DE MANEJO DE RESALTADO DE LETRA
 * OBJETIVO: Manejar las distintas maneras de resaltar la letra
 * ENTRADAS: Un octeto con el estado actual de resaltado
 * SALIDAS:  El nuevo estado o bien la respuesta a la consulta
 * NOTAS:    Cada bit del octeto maneja un modo de resaltado:
 *           Bit 0 -> Negrita
 *           Bit 1 -> Itlica
 *           Bit 2 -> Versalita
 *           Bit 3 -> Subndice
 *           Bit 4 -> Superndice
 *------------------------------------------------------------------*/

#define Negrita(e)       (((e)&1)>0)
#define PonNegrita(e)    ((e)|=(octeto)1)
#define QuitaNegrita(e)  ((e)&=(octeto)254)

#define Italica(e)       (((e)&2)>0)
#define PonItalica(e)    ((e)|=(octeto)2)
#define QuitaItalica(e)  ((e)&=(octeto)253)

#define Versalita(e)       (((e)&4)>0)
#define PonVersalita(e)    ((e)|=(octeto)4)
#define QuitaVersalita(e)  ((e)&=(octeto)251)

#define Subindice(e)       (((e)&8)>0)
#define PonSubindice(e)    ((e)|=(octeto)8)
#define QuitaSubindice(e)  ((e)&=(octeto)247)

#define Superindice(e)       (((e)&16)>0)
#define PonSuperindice(e)    ((e)|=(octeto)16)
#define QuitaSuperindice(e)  ((e)&=(octeto)239)

/*--------------------------------------------------------------------
 * Definicin de variables globales
 *------------------------------------------------------------------*/

cadena ExtensionQuill[4];
entero LongTexto;

cadena   Digito = "0123456789ABCDEF";
octeto   CarExt[4] = {92, 39, 0, 0};   /*  \'00  */
octeto   TamanoSubSuper=58;
octeto   PosicionSubSuper=33;

/*--------------------------------------------------------------------
 * Programa principal
 *------------------------------------------------------------------*/
void main (Narg, Arg)
int   Narg;
char *Arg[];
  {
  enum { SALIDA_ESC, SALIDA, EXPLICA, FICHERO, OPCIONES };

  static cadena Principal[] = {">Salida", "E>xplica", ">Fichero",
                               ">Opciones", NIL };
  entero Opcion;
  logico Sigue=SI;
  cadena Nombre;

  Pan_Define (PAN_TEXTO);
  Prg_Presenta ("Quill a RTF", "1.1", "Pedro Reina", "Marzo 1998");
  Cdr_Caja (CDR_SIMPLE,1,0,3,79,NEGRO,VERDE);

  Cad_Copia (ExtensionQuill,"doc");

  if ( Narg == 1 )
    { Nombre = Fch_Nombre ( "QuillRTF","cnf"); }
  else
    { Nombre = Cad_Duplica ( Arg[1] ); }

  if ( Fch_Existe (Nombre) )  { LeeConfig (Nombre); }
  Cad_Destruye (Nombre);

  Opcion = FICHERO;
  while ( Sigue )
    {
    Opcion = Men_Horizontal (2,2,77,Principal,Opcion);
    switch ( Opcion )
      {
      case SALIDA_ESC:
      case SALIDA:
           if ( Usr_Consulta ("Quieres terminar el programa?") )
             { Sigue = NO; }
           Opcion = SALIDA;
           break;
      case EXPLICA:    Explica();     break;
      case FICHERO:    Fichero();     break;
      case OPCIONES:   Opciones();    break;
      }
    LimpiaZonaTrabajo();
    }

  Pan_Cierra();
  }

/*--------------------------------------------------------------------
 * Definicin de funciones
 *------------------------------------------------------------------*/

/*--------------------------------------------------------------------
 * FUNCION:  Explica()
 * OBJETIVO: Dirigir las explicaciones
 *------------------------------------------------------------------*/
void Explica()
  {
  enum { SALIDA_ESC, OBJETIVO, PROGRAMA, AUTOR };
  static contador MaxOpcion = 3;

  static cadena Menu[] = {">Objetivo", ">Programa", ">Autor", NIL };

  static cadena Presentacion[] = {
    "Puedes pedir explicacin sobre diferentes aspectos",
    "de este programa. Elige el que desees y pulsa ESC",
    "cuando hayas terminado.",
    NIL };

  logico Sigue = SI;
  entero   Opcion = 1;
  contador i;

  while ( Sigue )
    {
    LimpiaZonaTrabajo();
    Pan_Color (NEGRO,BLANCO);
    for ( i=0 ; Presentacion[i] ; i++ )
      { Pan_PonTexto (6+i,15,Presentacion[i]); }

    Opcion = Men_Vertical (11,34,14,50,Menu,Opcion);
    LimpiaZonaTrabajo();
    switch ( Opcion )
      {
      case SALIDA_ESC:  Sigue = NO;         break;
      case OBJETIVO:    ExplicaObjetivo();  break;
      case PROGRAMA:    ExplicaPrograma();  break;
      case AUTOR:       ExplicaAutor();     break;
      }
    Opcion++;
    if ( Opcion > MaxOpcion )  { Opcion = 1; }
    }
  }

/*--------------------------------------------------------------------
 * FUNCION:  ExplicaProblema()
 * OBJETIVO: Explicar en qu consiste el problema
 *------------------------------------------------------------------*/
void ExplicaObjetivo()
  {
  static cadena Explicacion[] = {
    "Este programa convierte un fichero Quill (de PSION)",
    "en un fichero Rich Text Format (RTF), que se puede",
    "importar directamente cualquier programa que maneje",
    "correctamente ficheros RTF.",
    "",
    "Se conserva la letra negrita, los subndices y",
    "superndices; la letra subrayada se convierte en",
    "itlica.",
    "",
    "El programa se puede configurar desde un fichero.",
    NIL };

  contador i;

  Pan_Color (NEGRO,BLANCO);
  for ( i=0 ; Explicacion[i] ; i++ )
    { Pan_PonTexto (5+i,10,Explicacion[i]); }

  Usr_PulsaUnaTecla ("");
  }

/*--------------------------------------------------------------------
 * FUNCION:  ExplicaPrograma()
 * OBJETIVO: Explicar el funcionamiento del programa
 *------------------------------------------------------------------*/
void ExplicaPrograma()
  {
  static cadena Explicacion[] = {
    "El programa est dirigido por mens. El principal es el",
    "men horizontal que siempre est presente en la parte",
    "superior de la pantalla.",
    "",
    "Con la opcin Fichero puedes convertir un fichero de",
    "Quill a RTF.",
    "",
    "En Opciones se pueden definir tres cosas: si el sonido",
    "est encendido o apagado y el tamao y la posicin de las",
    "letras que vayan en subndice o superndice, que se mide",
    "porcentualmente en relacin con la letra base.",
    "",
    "Para ms informacin, lee el fichero TXT que acompaa",
    "al programa.",
    NIL };

  contador i;

  Pan_Color (NEGRO,BLANCO);
  for ( i=0 ; Explicacion[i] ; i++ )
    { Pan_PonTexto (5+i,10,Explicacion[i]); }

  Usr_PulsaUnaTecla ("");
  }

/*--------------------------------------------------------------------
 * FUNCION:  ExplicaAutor()
 * OBJETIVO: Explicar quin es el autor y cmo contactar con l
 *------------------------------------------------------------------*/
void ExplicaAutor()
  {
  static cadena Mensaje[] = {
    "Este programa es de Dominio Pblico. Lo puedes usar, distribuir",
    "y modificar como desees.",
    "",
    "Ha sido escrito en C usando el sistema de programacin Olimpo;",
    "si no dispones de la versin 2.0 de Olimpo, puedes cargarla",
    "directamente desde mi sede web o pedrmela si no dispones de",
    "acceso a Internet.",
    "",
    "Si tienes cualquier duda o consulta, cuntamela e intentar ayudarte.",
    "",
    "Mis datos:",
    "",
    "Pedro Reina",
    "c/ Marquesa de Argeso, 4      Telfono: 91 565 17 59",
    "28019 Madrid                   Correo electrnico: pedro@anit.es",
    "Espaa                         Web: www.anit.es/pedro",
    NIL };

  octeto i;

  Pan_Color (NEGRO,BLANCO);
  for ( i=0 ; Mensaje[i] ; i++ )
    { Pan_PonTexto (5+i,5,Mensaje[i]); }

  Usr_PulsaUnaTecla ("");
  }

/*--------------------------------------------------------------------
 * FUNCION:  Fichero()
 * OBJETIVO: Dirigir la conversin de ficheros
 * ENTRADAS: Ninguna
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void Fichero()
  {
  enum { SALIDA_ESC, CONVIERTE, DIRECTORIO, MASCARA, EXTENSN };

  static cadena MenuFichero[] = {">Convierte", ">Directorio",
                                 ">Mscara", ">Extensin", NIL };

  static char Directorio [LONG_DIRECTORIO+1];
  static char Nombre [LONG_NOMBRE+1];
  static char Mascara [LONG_MASCARA+1] = "*";
  static char Extension [LONG_EXTENSION+1];
  static logico Primero=SI;

  cadena   Aux;
  entero   Opcion;
  contador Fil=4, Col=20, Ancho=40, Alto=5;
  logico   Sigue = SI, Refresca = SI;

  if ( Primero )
    {
    Cad_Copia (Extension, ExtensionQuill);
    Primero = NO;
    }

  Cdr_Caja (CDR_SIMPLE,Fil,Col,Fil+Alto,Col+Ancho,NEGRO,VERDE);

  Opcion = CONVIERTE;
  while ( Sigue )
    {
    if ( Refresca )
      {
      Pan_Borra (NEGRO,Fil+2,Col+15,Fil+Alto-1,Col+Ancho-1);
      Pan_Color (NEGRO,VERDE);
      Pan_PonTexto (Fil+2,Col+15,Directorio);
      Pan_PonTexto (Fil+3,Col+15,Mascara);
      Pan_PonTexto (Fil+4,Col+15,Extension);
      Refresca = NO;
      }

    Opcion = Men_Vertical (Fil+1,Col+1,Fil+Alto-1,Col+13,MenuFichero,Opcion);
    switch ( Opcion )
      {
      case SALIDA_ESC:  Sigue = NO;      break;

      case CONVIERTE:   Aux = EligeUnNombre (Directorio, Mascara,
                                             Extension);
                        if ( Aux )
                          {
                          if ( Cad_Longitud(Aux) > LONG_NOMBRE )
                            { Aux[LONG_NOMBRE] = NULO; }
                          Cad_Copia (Nombre, Aux);
                          Cad_Destruye (Aux);
                          TrataFichero (Directorio,Nombre,Extension);
                          }
                        break;

      case DIRECTORIO:  Aux = PideDirectorio(Directorio);
                        if ( Tec_Ultima() != TEC_ESC )
                          {
                          Cad_Copia (Directorio,Aux);
                          Refresca = SI;
                          }
                        Cad_Destruye (Aux);
                        break;

      case EXTENSN:     Aux = PideExtension(Extension);
                        if ( Tec_Ultima() != TEC_ESC )
                          {
                          Cad_Copia (Extension,Aux);
                          Refresca = SI;
                          }
                        Cad_Destruye (Aux);
                        break;

      case MASCARA:     Aux = PideMascara(Mascara);
                        if ( Tec_Ultima() != TEC_ESC )
                          {
                          Cad_Copia (Mascara,Aux);
                          Refresca = SI;
                          }
                        Cad_Destruye (Aux);
                        break;
      }
    LimpiaZonaPequena();
    }
  }

/*--------------------------------------------------------------------
 * FUNCION:  TrataFichero()
 * OBJETIVO: Preparar la conversin.
 * ENTRADAS: Directorio, nombre y extensin del fichero
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void TrataFichero (Dir,Nom,Ext)
cadena Dir,Nom,Ext;
  {
  cadena  NomQuill, NomRtf;
  fichero FichQuill, FichRtf;

  NomQuill = NombreCompleto(Dir,Nom,Ext);
  NomRtf = NombreCompleto(Dir,Nom,"rtf");

  if ( (FichQuill = Fch_AbreLeer (NomQuill,FCH_BINARIO)) )
    {
    if ( CompruebaQuill (FichQuill) )
      {
      if ( (FichRtf = Fch_AbreGrabar (NomRtf,FCH_BINARIO)) )
        {
        LimpiaZonaPequena();
        Convierte (FichQuill, FichRtf);
        Fch_Cierra (FichRtf);
        }
      }
    Fch_Cierra (FichQuill);
    Usr_PulsaUnaTecla ("Conversin terminada");
    }

  Cad_Destruye (NomQuill);
  Cad_Destruye (NomRtf);
  }

/*--------------------------------------------------------------------
 * FUNCION:  CompruebaQuill()
 * OBJETIVO: Comprobar que un fichero tiene formato Quill y
 *           dejarlo en la posicin adecuada para tratarlo
 * ENTRADAS: El fichero Quill, abierto y en posicin 0
 * SALIDAS:  Lgica
 * ALGORITMO:
 *      Los 2 primeros octetos son la longitud de la cabecera
 *      Los 8 siguientes son la cadena "vrm1qdf0"
 *      Los 4 siguientes son la posicin en que acaba el texto
 *      Cuando termina la cabecera comienza el texto; los dos primeros
 *                prrafos son la cabecera y el pie del documento
 *------------------------------------------------------------------*/
logico CompruebaQuill (Fichero)
fichero Fichero;
  {
  logico   Error=NO, VistoFinParrafo;
  octeto   Dato[20], Vez;
  entero   i;
  contador LongCabecera;

  if ( !Fch_LeeOcteto (Fichero,Dato,20) )  { Error = SI; }

  if ( !Error )
    {
    LongCabecera = 256L * Dato[0] + Dato[1];
    LongTexto = 256L * 256L * 256L * Dato[10] + 256L * 256L * Dato[11] +
                256L * Dato[12] + Dato[13] - LongCabecera - 1;

    Dato[10] = NULO;
    if ( !Cad_Igual (Dato+2, "vrm1qdf0") )
      { Error = SI; }
    }

  Error = !Fch_Coloca (Fichero, LongCabecera);

  if ( !Error )
    {
    for ( Vez = 0 ; Vez<2 && !Error ; Vez++ )
      {
      VistoFinParrafo = NO;
      while ( !VistoFinParrafo && !Error )
        {
        if ( !Fch_LeeOcteto (Fichero,Dato,1) )  { Error = SI; }
        else
          {
          LongTexto--;
          if ( Dato[0] == 0 )  { VistoFinParrafo = SI; }
          }
        }
      }
    }

  if ( Error )
    { Usr_Avisa ("El fichero no tiene formato correcto"); }

  return ( !Error );
  }

/*--------------------------------------------------------------------
 * FUNCION:  Convierte()
 * OBJETIVO: Realizar la conversin
 * ENTRADAS: Los ficheros Quill y RTF, abiertos y en posicin correcta
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void Convierte (Quill, RichTex)
fichero Quill;
fichero RichTex;
  {
  entero   i, j;
  octeto   Dato, Estado=0;
  cadena   Aux;
  contador Tamano=12;  /* Lo tomo por defecto */
  logico   Visto;
  caracter Orden[50];

  Usr_Informa ("Convirtiendo");
  Cdr_Caja (CDR_SIMPLE,12,20,14,60,NEGRO,VERDE);
  Pan_Color (NEGRO,VERDE);
  Pan_PonTexto (13,23,"Octeto en conversin:");
  Pan_Tinta (BLANCO);
  Pan_CursorVisible (NO);

  Rtf_EscribeCadena (RichTex,"{\\rtf1\\pc\\deff0");
  Rtf_NuevaLinea (RichTex);

  for ( i=0 ; i<LongTexto && Fch_LeeOcteto (Quill,&Dato,1) ; i++ )
    {
    Pan_PonEntero (13,45,i,1);

    if ( Dato>=32 && Dato<=127 )  { Fch_EscribeOcteto (RichTex,&Dato,1); }
    else
      {
      switch ( Dato )
        {
        case GUIONOPCIONAL: break;

        case NEGRITA:
                        if ( Negrita(Estado) )
                          {
                          QuitaNegrita (Estado);
                          Rtf_EscribeCadena (RichTex,"\\b0 ");
                          }
                        else
                          {
                          PonNegrita (Estado);
                          Rtf_EscribeCadena (RichTex,"\\b1 ");
                          }
                        break;

        case SUBRAYADO:
                        if ( Italica(Estado) )
                          {
                          QuitaItalica (Estado);
                          Rtf_EscribeCadena (RichTex,"\\i0 ");
                          }
                        else
                          {
                          PonItalica (Estado);
                          Rtf_EscribeCadena (RichTex,"\\i1 ");
                          }
                        break;

        case SUPERINDICE:
                        if ( Superindice(Estado) )
                          { QuitaSuperindice (Estado); }
                        else
                          { PonSuperindice (Estado); }

                        ExplicaEstado (Estado,Tamano,Orden);
                        Rtf_EscribeCadena (RichTex,Orden);
                        break;

        case SUBINDICE:
                        if ( Subindice(Estado) )
                          { QuitaSubindice (Estado); }
                        else
                          { PonSubindice (Estado); }

                        ExplicaEstado (Estado,Tamano,Orden);
                        Rtf_EscribeCadena (RichTex,Orden);
                        break;

        case FINPARRAFO: Rtf_NuevaLinea (RichTex);
                         Estado = 0;
                         Rtf_EscribeCadena (RichTex,"\\par");
                         ExplicaEstado (Estado,Tamano,Orden);
                         Rtf_EscribeCadena (RichTex,Orden);
                         break;

        default: Rtf_EscribeCaracter (RichTex,Dato);
        }
      }
    }

  Rtf_EscribeCadena (RichTex,"}");
  }

/*--------------------------------------------------------------------
 * FUNCION:  ExplicaEstado()
 * OBJETIVO: Escribir en una cadena el estado de resaltado de letra
 * ENTRADAS: El estado, la altura de la fuente y una cadena donde
 *           dejar la informacin. Tambin se usan dos variables globales
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void ExplicaEstado (Estado, Tamano, Cadena)
octeto   Estado;
contador Tamano;
cadena   Cadena;
  {
  caracter Aux[20];
  cadena   Pos;

  Tamano *= 2;

  if ( Subindice(Estado) )         { Pos = "\\dn"; }
  else if ( Superindice(Estado) )  { Pos = "\\up"; }

  if ( Subindice(Estado) || Superindice(Estado) )
    {
    sprintf (Aux, "%s%d\\fs%d ", Pos, PosicionSubSuper*Tamano/100,
                                      TamanoSubSuper*Tamano/100);
    }
  else  { sprintf (Aux," "); }

  sprintf (Cadena, "\\plain\\b%d\\i%d\\fs%d%s",
           Negrita(Estado),Italica(Estado),Tamano,Aux);
  }

/*--------------------------------------------------------------------
 * FUNCION:  EligeUnNombre()
 * OBJETIVO: Mostrar al usuario una lista con los ficheros que puede leer
 *           y devolver el que elija
 * ENTRADAS: Directorio, mscara para el nombre, extensin del fichero y
 *           un nmero para retocar las dimensiones de la caja
 * SALIDAS:  Una cadena con el nombre (sin directorio ni extensin) o NIL
 * NOTA:     La cadena devuelta hay que destruirla cuando no sea necesaria
 *------------------------------------------------------------------*/
cadena EligeUnNombre (Dir,Nom,Ext)
cadena Dir,Nom,Ext;
  {
  contador Fil=10, Col=25, Ancho=26, Alto=11;
  cadena   Mascara, Mensaje, Respuesta = NIL;
  cadena   MenuPosible [MAX_FICHERO+1];
  lista    Posible, L;
  entero   Elegido;
  contador i;
  logico   Sigue;

  LimpiaZonaPequena();
  Cdr_Caja (CDR_SIMPLE,Fil,Col,Fil+Alto,Col+Ancho,NEGRO,VERDE);

  Mascara = NombreCompleto(Dir,Nom,Ext);
  Usr_Informa ("Buscando ficheros");
  Posible = Fch_ListaNombre (Mascara);
  if ( Posible )
    {
    for ( i=0 , L=Posible ; L && i<MAX_FICHERO ; L=Lis_Siguiente(L) )
      { MenuPosible[i++] = Lis_Contenido(L); }
    MenuPosible[i] = NIL;

    Pan_Color (NEGRO,VERDE);
    Pan_PonTexto (Fil+1,Col+4,"Elije el fichero:");
    if ( Elegido =
         Men_Vertical (Fil+3,Col+4,Fil+Alto-1,Col+20,MenuPosible,(entero)1) )
      {
      Sigue = SI;
      for ( i=Cad_Longitud(MenuPosible[Elegido-1]) ; i>=0 && Sigue ; i-- )
        {
        if ( MenuPosible[Elegido-1][i] == Fch_Separador() )
          {
          MenuPosible[Elegido-1][i] = NULO;
          Sigue = NO;
          }
        }
      Respuesta = Cad_Duplica (MenuPosible[Elegido-1]);
      }
    Lis_Destruye (Posible);
    }

  else
    {
    Mensaje = Cad_Une ("No se encontrado ningn fichero ","\"",Mascara,
                       "\"", NIL);
    Usr_PulsaUnaTecla (Mensaje);
    Cad_Destruye (Mensaje);
    }

  Cad_Destruye (Mascara);
  return ( Respuesta );
  }

/*--------------------------------------------------------------------
 * FUNCION:  NombreCompleto()
 * OBJETIVO: Devolver el nombre completo de un fichero
 * ENTRADAS: Directorio, nombre y extensin del fichero
 * SALIDAS:  Una cadena con el nombre completo
 * NOTA:     La cadena hay que destruirla cuando no sea necesaria
 *------------------------------------------------------------------*/
cadena NombreCompleto (Dir,Nom,Ext)
cadena Dir,Nom,Ext;
  {
  cadena Respuesta, Aux;

  Aux = Fch_Nombre (Nom,Ext);
  Respuesta = Cad_Une (Dir,Aux,NIL);
  Cad_Destruye (Aux);

  return ( Respuesta );
  }

/*--------------------------------------------------------------------
 * FUNCION:  PideDirectorio()
 * OBJETIVO: Pedir al usuario un directorio
 * ENTRADAS: El directorio propuesto
 * SALIDAS:  El nuevo directorio
 * NOTA:     La cadena hay que destruirla cuando no sea necesaria
 *------------------------------------------------------------------*/
cadena PideDirectorio (Dir)
cadena Dir;
  {
  static contador Fil=13, Col=22, Ancho=36, Alto=6;

  Cdr_Caja (CDR_SIMPLE,Fil,Col,Fil+Alto,Col+Ancho,NEGRO,VERDE);

  Pan_Color (NEGRO,VERDE);
  Pan_PonTexto (Fil+2,Col+6,"Escribe el directorio:");
  return ( Usr_Texto (Dir,LONG_DIRECTORIO,Fil+4,Col+6,NEGRO,BLANCO) );
  }

/*--------------------------------------------------------------------
 * FUNCION:  PideExtension()
 * OBJETIVO: Pedir al usuario una extensin
 * ENTRADAS: La extensin propuesta
 * SALIDAS:  La nueva extensin
 * NOTA:     La cadena hay que destruirla cuando no sea necesaria
 *------------------------------------------------------------------*/
cadena PideExtension (Ext)
cadena Ext;
  {
  static contador Fil=13, Col=23, Ancho=34, Alto=6;

  Cdr_Caja (CDR_SIMPLE,Fil,Col,Fil+Alto,Col+Ancho,NEGRO,VERDE);

  Pan_Color (NEGRO,VERDE);
  Pan_PonTexto (Fil+2,Col+6,"Escribe la extensin:");
  return ( Usr_Texto (Ext,LONG_EXTENSION,Fil+4,Col+8,NEGRO,BLANCO) );
  }

/*--------------------------------------------------------------------
 * FUNCION:  PideMascara()
 * OBJETIVO: Pedir al usuario una mscara
 * ENTRADAS: La mscara propuesta
 * SALIDAS:  La nueva mscara
 * NOTA:     La cadena hay que destruirla cuando no sea necesaria
 *------------------------------------------------------------------*/
cadena PideMascara (Mas)
cadena Mas;
  {
  static contador Fil=13, Col=24, Ancho=32, Alto=6;

  Cdr_Caja (CDR_SIMPLE,Fil,Col,Fil+Alto,Col+Ancho,NEGRO,VERDE);

  Pan_Color (NEGRO,VERDE);
  Pan_PonTexto (Fil+2,Col+6,"Escribe la mscara:");
  return ( Usr_Texto (Mas,LONG_MASCARA,Fil+4,Col+8,NEGRO,BLANCO) );
  }

/*--------------------------------------------------------------------
 * FUNCION:  PideNombre()
 * OBJETIVO: Pedir al usuario un nombre de fichero
 * ENTRADAS: El nombre propuesto y la mxima longitud admitida
 * SALIDAS:  El nuevo nombre
 * NOTA:     La cadena hay que destruirla cuando no sea necesaria
 *------------------------------------------------------------------*/
cadena PideNombre (Nom,MaxLong)
cadena   Nom;
contador MaxLong;
  {
  static contador Fil=13, Col=22, Ancho=36, Alto=6;

  Cdr_Caja (CDR_SIMPLE,Fil,Col,Fil+Alto,Col+Ancho,NEGRO,VERDE);

  Pan_Color (NEGRO,VERDE);
  Pan_PonTexto (Fil+2,Col+4,"Escribe el nombre del fichero:");
  return ( Usr_Texto (Nom,MaxLong,Fil+4,Col+6,NEGRO,BLANCO) );
  }

/*--------------------------------------------------------------------
 * FUNCION:  LeeConfig()
 * OBJETIVO: Leer un fichero de configuracin
 *------------------------------------------------------------------*/
void LeeConfig (Nombre)
cadena Nombre;
  {
  config Config;
  cadena Linea;
  char  *Etiqueta, *Aux, *Aux2;

  Usr_Informa ("Leyendo configuracin");

  if ( Config = Cnf_Abre (Nombre) )
    {
    while ( Linea = Cnf_Lee (Config) )
      {
      Cad_RecortaFinal (Linea);
      if ( Etiqueta = strtok (Linea, " ") )
        {

        if ( Cad_Igual (Etiqueta,"Sonido") )
          {
          if ( Aux = strtok (NULL, " ") )
            {
            if ( toupper(Aux[0]) == 'S' ) { Son_Enciende(); }
            else                          { Son_Apaga(); }
            }
          }

        else if ( Cad_Igual (Etiqueta,"TamaoSubSuper") )
          {
          if ( Aux = strtok (NULL, " ") )
            {
            TamanoSubSuper = atoi (Aux);
            TamanoSubSuper = Min (TamanoSubSuper,100);
            }
          }

        else if ( Cad_Igual (Etiqueta,"PosicinSubSuper") )
          {
          if ( Aux = strtok (NULL, " ") )
            {
            PosicionSubSuper = atoi (Aux);
            PosicionSubSuper = Min (PosicionSubSuper,100);
            }
          }

        } /* Fin if Etiqueta */
      Cad_Destruye (Linea);
      } /* Fin while Linea */
    } /* Fin if Cnf_Abre() */

  }

/*--------------------------------------------------------------------
 * FUNCION:  Rtf_EscribeCadena()
 * OBJETIVO: Escribir en un fichero RTF una cadena
 * ENTRADAS: El fichero y la cadena
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void Rtf_EscribeCadena (Fichero, Cadena)
fichero Fichero;
cadena  Cadena;
  {
  Fch_EscribeOcteto (Fichero,Cadena,Cad_Longitud(Cadena));
  }

/*--------------------------------------------------------------------
 * FUNCION:  Rtf_EscribeCaracter()
 * OBJETIVO: Escribir en un fichero RTF un octeto
 * ENTRADAS: El fichero y el octeto
 * SALIDAS:  Ninguna
 * NOTA:     Esta funcin es especfica para este programa, no es general
 *------------------------------------------------------------------*/
void Rtf_EscribeCaracter (Fichero,Octeto)
fichero Fichero;
octeto  Octeto;
  {
  if ( Octeto < 32 )  { Fch_EscribeOcteto (Fichero,&Octeto,1); }
  else
    {
    Octeto = Car_ConvierteQLaPC ((caracter)Octeto);
    CarExt[2] = Digito [Octeto / 16];
    CarExt[3] = Digito [Octeto % 16];
    Fch_EscribeOcteto (Fichero,CarExt,4);
    }
  }

/*--------------------------------------------------------------------
 * FUNCION:  Rtf_NuevaLinea()
 * OBJETIVO: Escribir en un fichero RTF el (los) carcter(es) de
 *           nueva lnea
 * ENTRADAS: El fichero
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void Rtf_NuevaLinea (Fichero)
fichero Fichero;
  {
  static octeto NL[2] = {13, 10};
#ifdef OLIMPO_PC
  Fch_EscribeOcteto (Fichero,NL,2);
#endif
#ifdef OLIMPO_QL
  Fch_EscribeOcteto (Fichero,NL+1,1);
#endif
  }

/*--------------------------------------------------------------------
 * FUNCION:  Opciones()
 * OBJETIVO: Seleccionar las opciones del programa
 * ENTRADAS: Ninguna
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void Opciones()
  {
  enum { SALIDA_ESC, SONIDO, TAMANO, POSICION};

  static cadena MenuOpc[] = {">Sonido", ">Tamao Sub/Sup",
                             ">Posicin Sub/Sup", NIL };

  entero   Opcion;
  contador Fil=4, Col=48, Ancho=31, Alto=4;
  logico   Sigue = SI;

  Cdr_Caja (CDR_SIMPLE,Fil,Col,Fil+Alto,Col+Ancho,NEGRO,VERDE);

  Opcion = SONIDO;
  while ( Sigue )
    {
    Pan_Color (NEGRO,VERDE);

    if ( Son_Estado() ) { Pan_PonTexto (Fil+1,Col+20,"Encendido"); }
    else                { Pan_PonTexto (Fil+1,Col+20,"Apagado  "); }
    Pan_PonEntero (Fil+2,Col+20,TamanoSubSuper,1);
    Pan_PonEntero (Fil+3,Col+20,PosicionSubSuper,1);

    Opcion = Men_Vertical (Fil+1,Col+1,Fil+Alto-1,Col+19,MenuOpc,Opcion);
    switch ( Opcion )
      {
      case SALIDA_ESC:    Sigue = NO;      break;
      case SONIDO:        Son_Cambia();    break;
      case TAMANO:        TamanoSubSuper = PideTamano (TamanoSubSuper);
                          break;
      case POSICION:      PosicionSubSuper = PidePosicion (PosicionSubSuper);
                          break;
      }
    }
  }

/*--------------------------------------------------------------------
 * FUNCION:  PideTamano()
 * OBJETIVO: Pedir al usuario el tamao de sub y superndice
 * ENTRADAS: El tamao propuesto
 * SALIDAS:  El nuevo tamao
 *------------------------------------------------------------------*/
octeto PideTamano (Tamano)
octeto Tamano;
  {
  octeto Respuesta;
  static contador Fil=12, Col=30, Ancho=19, Alto=4;

  Cdr_Caja (CDR_DOBLE,Fil,Col,Fil+Alto,Col+Ancho,NEGRO,VERDE);

  Pan_Color (NEGRO,VERDE);
  Pan_PonTexto (Fil+1,Col+2,"Elige el tamao:");
  Respuesta = Usr_Entero (Tamano, 4, 1, 100, Fil+3, Col+7, NEGRO, BLANCO);
  Pan_Borra (NEGRO,Fil,Col,Fil+Alto,Col+Ancho);

  return ( Respuesta );
  }

/*--------------------------------------------------------------------
 * FUNCION:  PidePosicion()
 * OBJETIVO: Pedir al usuario la posicin de sub y superndice
 * ENTRADAS: La posicin propuesta
 * SALIDAS:  El nuevo tamao
 *------------------------------------------------------------------*/
octeto PidePosicion (Posicion)
octeto Posicion;
  {
  octeto Respuesta;
  static contador Fil=12, Col=29, Ancho=21, Alto=4;

  Cdr_Caja (CDR_DOBLE,Fil,Col,Fil+Alto,Col+Ancho,NEGRO,VERDE);

  Pan_Color (NEGRO,VERDE);
  Pan_PonTexto (Fil+1,Col+2,"Elige la posicin:");
  Respuesta = Usr_Entero (Posicion, 4, 1, 100, Fil+3, Col+7, NEGRO, BLANCO);
  Pan_Borra (NEGRO,Fil,Col,Fil+Alto,Col+Ancho);

  return ( Respuesta );
  }