/*--------------------------------------------------------------------
 * FICHERO:  Movimien.c
 * OBJETIVO: Definir las funciones del objeto Movimiento
 * AUTOR:    Pedro Reina
 * FECHA:    D.8.3.1998
 *------------------------------------------------------------------*/

/*--------------------------------------------------------------------
 * Ficheros de cabecera
 *------------------------------------------------------------------*/

#include "Movimien.h"

/*--------------------------------------------------------------------
 * Definicin de constantes
 *------------------------------------------------------------------*/

#define MOV_LONGPARTIDA  81
#define MOV_LONGCABECERA 17

/*--------------------------------------------------------------------
 * Estructuras del almacenamiento de movimientos
 *------------------------------------------------------------------*/

contador Mov_Partida[MOV_MAX][4];   /* Los movimientos de la partida */
octeto   Mov_Info[MOV_LONGPARTIDA]; /* La partida para leer o grabar
                                       desde o en un fichero         */

/*--------------------------------------------------------------------
 * Estructura del fichero de partida:
 *   Todos los ficheros tienen 81 octetos
 *   Cabecera: 17 octetos
 *     Octetos de 0 a 13: La cadena "Solitario 1.1", terminada en NULO
 *     Octetos 14 y 15: Tiempo de la partida, el 14 es el ms significativo
 *     Octeto 16: Mov_Ultimo
 *   Movimientos: 64 octetos
 *     Cada movimiento tiene dos octetos:
 *       El primero indica la casilla origen
 *       El segundo indica la casilla destino
 *         Cada casilla tiene en los bits 0, 1 y 2 la columna y en los
 *                      3, 4 y 5 la fila
 *     Los movimientos siguientes a Mov_Ultimo se almacenan con casillas
 *                      A1, es decir, dos caracteres NULO
 *------------------------------------------------------------------*/

/*--------------------------------------------------------------------
 * Variables globales
 *------------------------------------------------------------------*/

contador Mov_Actual;  /* Indica el ltimo movimiento mostrado en
                         el tablero  */
contador Mov_Ultimo;  /* Indica el ltimo movimiento almacenado en
                         Mov_Partida */

contador Mov_Fil=5;  /* Fila superior de la zona asignada     */
contador Mov_Col=2;  /* Columna izquierda de la zona aisgnada */

/*--------------------------------------------------------------------
 * Definicin de funciones
 *------------------------------------------------------------------*/

/*--------------------------------------------------------------------
 * FUNCION:  Mov_Inicia()
 * OBJETIVO: Poner a cero todos los movimientos
 * ENTRADAS: Ninguna
 * SALIDAS:  La variables globales se modifican
 *------------------------------------------------------------------*/
void Mov_Inicia()
  {
  contador i,j;

  Mov_Actual = MOV_NINGUNO;
  Mov_Ultimo = MOV_NINGUNO;

  for ( i=0 ; i<MOV_MAX ; i++ ) {
  for ( j=0 ; j<4       ; j++ ) {
    Mov_Partida[i][j] = 0;
    }}
  }

/*--------------------------------------------------------------------
 * FUNCION:  Mov_DibujaMarco()
 * OBJETIVO: Dibujar las lneas y mensajes que expresan los movimientos
 * ENTRADAS: Ninguna
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void Mov_DibujaMarco()
  {
  Cdr_Caja (CDR_SIMPLE,Mov_Fil,Mov_Col,Mov_Fil+15,Mov_Col+16,NEGRO,ROJO);
  Pan_Color (NEGRO,VERDE);
  Pan_PonTexto (Mov_Fil+1,Mov_Col+3,"Movimientos");
  }

/*--------------------------------------------------------------------
 * FUNCION:  Mov_ActualizaPantalla()
 * OBJETIVO: Escribir en la pantalla los movimientos que quepan, terminando
 *           en el ltimo realizado
 * ENTRADAS: Ninguna
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void Mov_ActualizaPantalla()
  {
  contador Primero, Ultimo, i, Pos, Caben=11;

  Pan_Color (NEGRO,BLANCO);
  Pan_Borra (NEGRO,Mov_Fil+2,Mov_Col+1,Mov_Fil+14,Mov_Col+15);

  if ( Mov_Actual == MOV_NINGUNO )
    { Pan_PonTexto (Mov_Fil+3,Mov_Col+3,"Ninguno"); }
  else
    {
    Primero = Max (0,Mov_Actual-Caben);
    Ultimo = Mov_Actual;
    for ( i=Primero , Pos=Mov_Fil+3 ; i<=Ultimo ; i++ , Pos++ )
      { Mov_EscribeMovimiento (i,Pos); }
    }
  }

/*--------------------------------------------------------------------
 * FUNCION:  Mov_EscribeMovimiento()
 * OBJETIVO: Escribir en la pantalla un movimiento
 * ENTRADAS: El nmero del movimiento y la fila
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void Mov_EscribeMovimiento (Num,Fil)
  {
  Pan_PonEntero (Fil,Mov_Col+3,Num,2);
  Pan_Texto (". ");
  Mov_EscribeCasilla (Mov_Partida[Num][0],Mov_Partida[Num][1]);
  if ( Num )
    {
    Pan_Texto (" - ");
    Mov_EscribeCasilla (Mov_Partida[Num][2],Mov_Partida[Num][3]);
    }
  }

/*--------------------------------------------------------------------
 * FUNCION:  Mov_EscribeCasilla()
 * OBJETIVO: Escribir en la pantalla una casilla
 * ENTRADAS: Fila y columna de la casilla
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void Mov_EscribeCasilla (Fil,Col)
  {
  Pan_Caracter ('A'+Fil);
  Pan_Caracter ('1'+Col);
  }

/*--------------------------------------------------------------------
 * FUNCION:  Mov_Almacena()
 * OBJETIVO: Memorizar un movimiento. Esto hace que se pierdan los siguientes
 * ENTRADAS: Un vector indicando el movimiento actual
 * SALIDAS:  La partida y las variables globales quedan modificadas
 * NOTA:     Si la casilla destino es 0,0 (posicin A1), se interpreta que
 *           es el movimiento cero
 *------------------------------------------------------------------*/
void Mov_Almacena (Movimiento)
contador Movimiento[4];
  {
  contador i;

  for ( i=0 ; i<4 ; i++ ) { Mov_Partida[Mov_Actual+1][i] = Movimiento[i]; }

  Mov_IncrementaActual();
  }

/*--------------------------------------------------------------------
 * FUNCION:  Mov_Anterior()
 * OBJETIVO: Pasar al movimiento anterior al actual, si es posible
 * ENTRADAS: Un vector donde dejar el movimiento actual y el nmero de
 *           movimiento mnimo que se admite devolver
 * SALIDAS:  Lgica, indicando si ha sido posible.
 *           La variable Mov_Actual puede quedar modificada
 * NOTA:     No se admite volver a la situacin inicial, el movimiento
 *           cero no se devuelve
 *------------------------------------------------------------------*/
logico Mov_Anterior (Movimiento,Num)
contador Movimiento[4], Num;
  {
  logico   Respuesta;
  contador i;

  if ( Mov_Actual >= Num )
    {
    for ( i=0 ; i<4 ; i++ )
      { Movimiento[i] = Mov_Partida[Mov_Actual][i]; }
    Mov_Actual--;
    Respuesta = SI;
    }
  else
    { Respuesta = NO; }

  return ( Respuesta );
  }

/*--------------------------------------------------------------------
 * FUNCION:  Mov_Siguiente()
 * OBJETIVO: Pasar al movimiento siguiente al actual, si es posible
 * ENTRADAS: Un vector donde dejar el movimiento siguiente
 * SALIDAS:  Lgica, indicando si ha sido posible.
 *           La variable Mov_Actual puede quedar modificada
 *------------------------------------------------------------------*/
logico Mov_Siguiente (Movimiento)
contador Movimiento[4];
  {
  logico   Respuesta;
  contador i;

  if ( Mov_Actual != Mov_Ultimo )
    {
    Mov_Actual++;
    for ( i=0 ; i<4 ; i++ )
      { Movimiento[i] = Mov_Partida[Mov_Actual][i]; }
    Respuesta = SI;
    }
  else
    { Respuesta = NO; }

  return ( Respuesta );
  }

/*--------------------------------------------------------------------
 * FUNCION:  Mov_GrabaPartida()
 * OBJETIVO: Grabar una partida en un fichero
 * ENTRADAS: El nombre completo del fichero y el tiempo de partida
 * SALIDAS:  El fichero puede quedar modificado
 *------------------------------------------------------------------*/
void Mov_GrabaPartida (Nombre,Tiempo)
cadena   Nombre;
contador Tiempo;
  {
  fichero  Salida;
  contador i;
  cadena   Mens;

  if ( Salida = Fch_AbreGrabar (Nombre,FCH_BINARIO) )
    {
    Mens = Cad_Une ("Grabando la partida en \"",Nombre,"\"",NIL);
    Usr_Informa (Mens);
    Cad_Destruye (Mens);

    Mem_Asigna (Mov_Info,0,MOV_LONGPARTIDA);
    Cad_Copia (Mov_Info,"Solitario 1.1");
    Mov_Info[14] = Tiempo / 256;
    Mov_Info[15] = Tiempo % 256;
    Mov_Info[16] = Mov_Ultimo;
    for ( i=0 ; i<= Mov_Ultimo ; i++ )
      {
      Mov_Info[MOV_LONGCABECERA+2*i]   = Mov_Compacta(i,0); /* Origen */
      Mov_Info[MOV_LONGCABECERA+2*i+1] = Mov_Compacta(i,2); /* Destino */
      }
    Fch_EscribeOcteto (Salida,Mov_Info,MOV_LONGPARTIDA);
    Fch_Cierra (Salida);
    Usr_PulsaUnaTecla ("Partida grabada");
    }
  }

/*--------------------------------------------------------------------
 * FUNCION:  Mov_Compacta()
 * OBJETIVO: Compactar en un octeto una casilla
 * ENTRADAS: El nmero de movimiento y el 0  2: origen o destino
 * SALIDAS:  Un octeto con el resultado
 *------------------------------------------------------------------*/
unsigned char Mov_Compacta (Num,Pos)
contador Num, Pos;
  {
  octeto Resultado;

  Resultado = Mov_Partida[Num][Pos] << 3; /* Fila */
  Resultado |= Mov_Partida[Num][Pos+1];   /* Columna */

  return (Resultado);
  }

/*--------------------------------------------------------------------
 * FUNCION:  Mov_LeePartida()
 * OBJETIVO: Leer una partida de un fichero
 * ENTRADAS: El nombre completo del fichero
 * SALIDAS:  El tiempo de la partida, o el cdigo de error
 *           La partida puede quedar modificada
 *------------------------------------------------------------------*/
contador Mov_LeePartida (Nombre)
cadena Nombre;
  {
  fichero  Entrada;
  contador i, Respuesta;
  cadena   Mens;

  if ( Entrada = Fch_AbreLeer (Nombre,FCH_BINARIO) )
    {
    Mens = Cad_Une ("Leyendo la partida de \"",Nombre,"\"",NIL);
    Usr_Informa (Mens);
    Cad_Destruye (Mens);

    Mem_Asigna (Mov_Info,0,MOV_LONGPARTIDA);
    Fch_LeeOcteto (Entrada,Mov_Info,MOV_LONGPARTIDA);
    Fch_Cierra (Entrada);

    if ( ! Cad_Igual (Mov_Info,"Solitario 1.1") )
      {
      Mens = Cad_Une ("El fichero \"",Nombre,"\""," no contiene una partida.",
                      NIL);
      Usr_Avisa (Mens);
      Cad_Destruye (Mens);
      Respuesta = MOV_ERROR;
      }
    else
      {
      Respuesta = 256 * Mov_Info[14] + Mov_Info[15];
      Mov_Ultimo = Mov_Info[16];
      for ( i=0 ; i<= Mov_Ultimo ; i++ )
        {
        Mov_Partida[i][0] = Mov_SacaFil (Mov_Info[MOV_LONGCABECERA+2*i]);
        Mov_Partida[i][1] = Mov_SacaCol (Mov_Info[MOV_LONGCABECERA+2*i]);
        Mov_Partida[i][2] = Mov_SacaFil (Mov_Info[MOV_LONGCABECERA+2*i+1]);
        Mov_Partida[i][3] = Mov_SacaCol (Mov_Info[MOV_LONGCABECERA+2*i+1]);
        }
      Usr_PulsaUnaTecla ("Partida leda");
      }
    }

  else { Respuesta = MOV_ERROR; }

  return ( Respuesta );
  }