//--------------------------------------------------------------
// Fichero:  geo3d.js
// Objetivo: Rutinas JavaScript de geometría 3D con WebGL
// Fecha:    S.26.3.2022
// Versión:  Pedro Reina
// Original: https://www.ibiblio.org/e-notes/webgl/polyhedra/dodecahedron.html
// Licencia: All scripts are free for any use
//           https://www.ibiblio.org/e-notes/webgl/webgl.htm
//--------------------------------------------------------------

//---------------------------------
// Variables globales
//---------------------------------

var mvMatrix = new CanvasMatrix4();
var rotMat = new CanvasMatrix4();
var nFaces = ind.length, normLoc, mvMatLoc;
var norm = [];

var gl, canvas, pi180 = 180/Math.PI,
  transl = -3, rTouch, fiTouch, idTouch0,
  xRot = yRot = zRot = xOffs = yOffs = drag = 0;

//---------------------------------
// Funciones
//---------------------------------

//--------------------------------------------------------------
// La función que inicia el sistema cuando se carga la página
AgregaEvento (window, 'load', webGLStart, false);

//--------------------------------------------------------------
// La función que reinicia el dibujo si se cambia de tamaño
AgregaEvento (window, 'resize', canvasresize, false);

//---------------------------------
// Funciones de manejo de WebGL
//---------------------------------

//--------------------------------------------------------------
// El inicio de WebGL
function webGLStart()
   {
   initGL();
   var size = Math.min(window.innerWidth, window.innerHeight) - 15;
   canvas.width = size;
   canvas.height = size;
   gl.viewport(0, 0, size, size);

   var prog = gl.createProgram();
   gl.attachShader(prog, getShader(gl,"shader-vs"));
   gl.attachShader(prog, getShader(gl,"shader-fs"));
   gl.linkProgram(prog);
   gl.useProgram(prog);

   var posLocation = gl.getAttribLocation(prog,"aPos");
   gl.enableVertexAttribArray(posLocation);
   var posBuffer = gl.createBuffer();
   gl.bindBuffer(gl.ARRAY_BUFFER, posBuffer);
   gl.bufferData(gl.ARRAY_BUFFER, new Float32Array(vertices), gl.STATIC_DRAW);
   gl.vertexAttribPointer(posLocation, 3, gl.FLOAT, false, 0, 0);

   var app = [];
   for(var i=0; i<nFaces; i++)
       { app = app.concat(ind[i]); }
   var indexBuffer = gl.createBuffer();
   gl.bindBuffer(gl.ELEMENT_ARRAY_BUFFER, indexBuffer);
   gl.bufferData(gl.ELEMENT_ARRAY_BUFFER, new Uint16Array(app),
                 gl.STATIC_DRAW);
   normLoc = gl.getUniformLocation(prog,"normal");
   for(var i=0; i<nFaces; i++)
     {
     var j = 3*ind[i][0];
     var v0 = vertices.slice(j,j+3);
     j = 3*ind[i][1];
     var v1 = vertices.slice(j,j+3);
     j = 3*ind[i][2];
     var v2 = vertices.slice(j,j+3);
     var nx = (v1[1]-v0[1])*(v2[2]-v0[2]) - (v2[1]-v0[1])*(v1[2]-v0[2]);
     var ny =-(v1[0]-v0[0])*(v2[2]-v0[2]) + (v2[0]-v0[0])*(v1[2]-v0[2]);
     var nz = (v1[0]-v0[0])*(v2[1]-v0[1]) - (v2[0]-v0[0])*(v1[1]-v0[1]);
     var n = Math.sqrt(nx*nx + ny*ny + nz*nz);
     norm[i] = [nx/n, ny/n, nz/n];
     }

   var prMatrix = new CanvasMatrix4();
   prMatrix.perspective(45, 1, .1, 100);
   gl.uniformMatrix4fv(gl.getUniformLocation(prog,"prMatrix"),
      false, new Float32Array(prMatrix.getAsArray()));
   rotMat.makeIdentity();
   mvMatLoc = gl.getUniformLocation(prog,"mvMatrix");

   gl.clearDepth(1.0);
   gl.enable(gl.DEPTH_TEST);
   gl.clearColor(.9, .9, 1, 1);

   transl = -8;
   rotMat.rotate(RotIni[0],RotIni[1],RotIni[2],RotIni[3]);
   drawScene();
   }

//--------------------------------------------------------------
// Preparación inicial de WebGL
function initGL()
   {
   canvas = document.getElementById("canvas");
   if (!window.WebGLRenderingContext)
     {
     alert("Este navegador no admite WebGL");
     return;
     }
   try { gl = canvas.getContext("experimental-webgl"); }
   catch(e) {}
   if (!gl)
     {
     alert("No puedo comenzar WebGL");
     return;}
   initEvents();
   }

//--------------------------------------------------------------
// Carga de los shaders
function getShader(gl,id)
   {
   var shaderScript = document.getElementById(id);
   var str = "";
   var k = shaderScript.firstChild;
   while ( k )
     {
     if ( k.nodeType == 3 ) str += k.textContent;
     k = k.nextSibling;
     }
   var shader;
   if ( shaderScript.type == "x-shader/x-fragment" )
       { shader = gl.createShader(gl.FRAGMENT_SHADER); }
   else if ( shaderScript.type == "x-shader/x-vertex" )
       { shader = gl.createShader(gl.VERTEX_SHADER); }
   else return null;
   gl.shaderSource(shader, str);
   gl.compileShader(shader);
   if (gl.getShaderParameter(shader, gl.COMPILE_STATUS) == 0)
      { alert(id + "\n" + gl.getShaderInfoLog(shader)); }
   return shader;
   }

//--------------------------------------------------------------
// Asignación de los eventos de ratón
function initEvents()
   {
   canvas.addEventListener('DOMMouseScroll', wheelHandler, false);
   canvas.addEventListener('mousewheel', wheelHandler, false);
   canvas.addEventListener('mousedown', mymousedown, false);
   canvas.addEventListener('mouseup', mymouseup, false);
   canvas.addEventListener('mousemove', mymousemove, false);
   canvas.addEventListener('touchstart', startTouch, false);
   canvas.addEventListener('touchmove', continueTouch, false);
   canvas.addEventListener('touchend', stopTouch, false);
   }

//--------------------------------------------------------------
// Cada vez que hay que dibujar la escena
function drawScene()
    {
    rotMat.rotate(xRot/5,1,0,0);
    rotMat.rotate(yRot/5,0,1,0);
    rotMat.rotate(zRot,0,0,1);
    yRot = xRot = zRot = 0;
    mvMatrix.load(rotMat);
    mvMatrix.translate(0, 0, transl);
    gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);
    gl.uniformMatrix4fv(mvMatLoc, false,
                        new Float32Array(mvMatrix.getAsArray()));
    var off = 0;
    for(var i=0; i<nFaces; i++)
      {
      gl.uniform3fv(normLoc, norm[i]);
      gl.drawElements(gl.TRIANGLE_FAN, ind[i].length, gl.UNSIGNED_SHORT, off);
      off += 2*ind[i].length;
      }
    gl.flush();
    }

//--------------------------------------------------------------
// Si nos cambian el tamaño de la ventana
function canvasresize()
    {
    var size = Math.min(window.innerWidth, window.innerHeight) - 15;
    canvas.width = size;
    canvas.height = size;
    gl.viewport(0, 0, size, size);
    drawScene();
    }

//---------------------------------
// Funciones de manejo del ratón
//---------------------------------

function startTouch(evt)
  {
  var evList = evt.touches;
  if(evList.length == 1)
    {
    xOffs = evList[0].pageX;  yOffs = evList[0].pageY;
    drag = 1;
    }
  else if(evList.length == 2)
    {
    idTouch0 = evList[0].identifier;
    var dx = evList[1].pageX - evList[0].pageX;
    var dy = evList[1].pageY - evList[0].pageY;
    rTouch = Math.sqrt(dx*dx + dy*dy);
    fiTouch = Math.atan2(dy, dx);
    drag = 2;
    }
  evt.preventDefault();
  }

function continueTouch(evt)
  {
  if(drag == 1)
    {
    var x = evt.touches[0].pageX,  y = evt.touches[0].pageY;
    yRot = x - xOffs;  xRot = y - yOffs;
    xOffs = x;  yOffs = y;
    drawScene();
    }
  else if(drag == 2)
    {
    var dx = evt.touches[1].pageX - evt.touches[0].pageX;
    var dy = evt.touches[1].pageY - evt.touches[0].pageY;
    var r = Math.sqrt(dx*dx + dy*dy);
    var fi;
    if( idTouch0 == evt.touches[0].identifier ) fi = Math.atan2(dy, dx);
    else fi = Math.atan2(-dy, -dx);
    transl *= rTouch / r;
    zRot = pi180*(fiTouch - fi);
    rTouch = r;  fiTouch = fi;
    drawScene();
    }
  }

function stopTouch()
  { drag = 0; }

function mymousedown(ev)
  {
  drag  = 1;
  xOffs = ev.clientX;  yOffs = ev.clientY;
  }

function mymouseup(ev)
  { drag  = 0; }

function mymousemove(ev)
  {
  if ( drag == 0 ) return;
  if ( ev.shiftKey ) {
    transl *= 1 + (ev.clientY - yOffs)/1000;
    zRot = (xOffs - ev.clientX)*.3; }
  else {
    yRot = - xOffs + ev.clientX;  xRot = - yOffs + ev.clientY; }
  xOffs = ev.clientX;   yOffs = ev.clientY;
  drawScene();
  }

function wheelHandler(ev)
  {
  var del = 1.1;
  if (ev.shiftKey) del = 1.01;
  var ds = ((ev.detail || ev.wheelDelta) > 0) ? del : (1 / del);
  transl *= ds;
  ev.preventDefault();
  drawScene();
  }

//---------------------------------
// Función de asignación de eventos
//---------------------------------

//--------------------------------------------------------------
// Función portable que añade un evento
function AgregaEvento (elemento, evType, funcion, useCapture)
  {
  // Manejador de eventos compatible con navegadores IE5+, NS6 y Mozilla
  // Autor: Scott Andrew
  if ( elemento.addEventListener )
    {
    elemento.addEventListener (evType, funcion, useCapture);
    return true;
    }
  else if ( elemento.attachEvent )
    {
    var r = elemento.attachEvent ('on' + evType, funcion);
    return r;
    }
  else
    { elemento ['on' + evType] = funcion; }
  }


