//--------------------------------------------------------------
// Fichero:  psl.js
// Objetivo: Rutinas JavaScript de Parejas de software libre
// Fecha:    X.29.7.2015
// Autor:    Pedro Reina <pedro@pedroreina.net>
// Licencia: CC0 1.0 Universal
//           http://creativecommons.org/publicdomain/zero/1.0/
//--------------------------------------------------------------

//---------------------------------
// Constantes
//---------------------------------

// El tamaño de cada celda
var TamanoCelda = 128;

//---------------------------------
// Variables globales
//---------------------------------

// Los elementos HTML
var BotonRueda;
var BotonNueva;
var BotonParar;
var CanvasDibujo;
var ContextoDibujo;
var ZonaProgreso;
var BarraProgreso;
var BotonInfo;
var BotonHistorial;
var BotonBorrar;
var BotonCreditos;
var BotonLicencia;
var BotonDescarga;
var BotonTecnicas;
var BotonCerrar1;
var BotonCerrar2;
var BotonCerrar3;
var BotonCerrar4;
var BotonCerrar5;
var BotonCerrar6;
var DivInfo;
var DivReloj;
var DivHistorial;
var DivListaHistorial;
var DivCreditos;
var DivLicencia;
var DivDescarga;
var DivTecnicas;
var ImagenVacia;
var ImagenBien;
var ImagenMal;

// Dimensiones del área de dibujo
var Anchura, Altura;

// Cuántos símbolos tenemos en total
var TotalSimbolo = 35;

// Cuántos símbolos forman un problema
var TotalProblema = 15;

// Los símbolos
var ImagenSimbolo = [];

// Lo que ocupan todas las imágenes que precargamos
var TotalTamano;

// La suma de los tamaños de imágenes cargadas hasta el momento
var CargadoActual;

// Cuántas filas y columnas caben en la zona de dibujo
MaxFil = 5;
MaxCol = 6;

// Origen de la zona de dibujo
var OrigenX = 5;
var OrigenY = 5;

// Si la partida está en juego o no
var EnJuego;

// El array que contiene la secuencia de símbolos que hay que resolver
var Problema = [];

// Cuántas parejas se han encontrado hasta el momento
var ParejasEncontradas;

// El tiempo que queda para terminar la partida
var TiempoRestante;

// La lista de celdas que ya se han averiguado
var CeldaVacia = [];

// La pareja de celdas acertadas que hay que borrar
var CeldaBien = [];

// La pareja de celdas falladas que hay que voltear
var CeldaMal = [];

// Coordenadas de la primera celda pulsada
var Fil1, Col1;

// La parte de jugada en la que estamos
var Tanda;

// Temporizadores de los cambios
var TemporizadorCambio = [];

// El manejador del temporizador principal
var Temporizador;

// Para controlar el tiempo
var Antes;
var Ahora;

//---------------------------------
// Funciones
//---------------------------------

//--------------------------------------------------------------
// Inicio del sistema
function Inicio()
  {
  // Obtenemos el contexto de la zona para dibujar
  ContextoDibujo = CanvasDibujo.getContext("2d");

  // Dimensiones de la zona para dibujar
  Anchura = ContextoDibujo.canvas.clientWidth;
  Altura = ContextoDibujo.canvas.clientHeight;

  // Borramos de gris la zona para dibujar
  ContextoDibujo.fillStyle = "#CCC";
  ContextoDibujo.fillRect (0, 0, Anchura, Altura);

  // Preparamos la zona del reloj
  EscribeReloj ("");
  }

//--------------------------------------------------------------
// Prepara la pantalla tras cargar todas las imágenes
function Prepara()
  {
  // Eliminamos la barra de progreso
  ZonaProgreso.style.display = "none";

  // Borramos de blanco la zona para dibujar
  ContextoDibujo.fillStyle = "#FFF";
  ContextoDibujo.fillRect (0, 0, Anchura, Altura);

  // Ponemos el símbolo de interrogación
  Rellena (ImagenInterrogacion);

  // Cambiamos el estado de los botones
  BotonNueva.style.display = "inline";
  BotonRueda.style.display = "none";

  // Mientras esperamos que el usuario comience la partida...
  Fil1 = Aleatorio(MaxFil);
  Col1 = Aleatorio(MaxCol);
  Temporizador = setInterval (Espera, 3000);
  }

//--------------------------------------------------------------
// Muestra aleatoriamente imágenes
function Espera()
  {
  // Borramos la posición anterior
  Pinta (Fil1, Col1, ImagenInterrogacion);

  // Buscamos una nueva posición
  Fil1 = Aleatorio(MaxFil);
  Col1 = Aleatorio(MaxCol);

  // Elegimos una imagen
  var ImagenTmp = ImagenSimbolo[Aleatorio(TotalSimbolo)];

  // Ponemos la nueva imagen
  Pinta (Fil1, Col1, ImagenTmp);
  }

//--------------------------------------------------------------
// Nueva partida
function Nueva()
  {
  // Anulamos el temporizador
  window.clearInterval (Temporizador);

  // Cambiamos el estado de los botones
  BotonNueva.style.display = "none";
  BotonParar.style.display = "inline";

  // El array para preparar la secuencia de símbolos
  var ProblemaTmp = [];
  for ( var i=0 ; i<TotalSimbolo ; i++ )
    { ProblemaTmp.push (i); }

  // Revolvemos aleatoriamente
  Revuelve (ProblemaTmp);

  // Nos quedamos con los primeros
  ProblemaTmp = ProblemaTmp.slice (0, TotalProblema);

  // Repetimos cada símbolo
  Problema = [];
  for ( var i=0 ; i<TotalProblema ; i++ )
    {
    Problema.push (ProblemaTmp[i]);
    Problema.push (ProblemaTmp[i]);
    }

  // Revolvemos aleatoriamente
  Revuelve (Problema);

  // Mostramos todos los símbolos tapados
  for ( var Fil=0 ; Fil<MaxFil ; Fil++ )
    {
    for ( var Col=0 ; Col<MaxCol ; Col++ )
      { Pinta (Fil, Col, ImagenInterrogacion); }
    }

  // Damos dos minutos de tiempo
  DivReloj.style.color = '#00C';
  EscribeReloj ("120");
  TiempoRestante = 121;

  // Cuando comenzamos todas las celdas están llenas
  CeldaVacia = [];

  // No hay celdas que cambiar
  CeldaBien = [];
  CeldaMal = [];

  // Cuando empezamos no hay ningún símbolo mostrado
  Tanda = 0;

  // Iniciamos las listas de temporizadores
  TemporizadorCambio = [];

  // Empieza la partida
  EnJuego = true;

  // Preparamos los tiempos
  Antes = 0;
  Ahora = Reloj();

  // Ponemos a cero el contador de parejas
  ParejasEncontradas = 0;

  // Activamos el temporizador principal cada 0.2 s
  Temporizador = setInterval (CambiaReloj, 200);
  }

//--------------------------------------------------------------
// Para la partida actual
function Parar()
  {
  // Terminó la partida
  EnJuego = false;

  // Eliminamos los temporizadores
  window.clearInterval (Temporizador);
  for ( var i=0 ; i<TemporizadorCambio.length ; i++ )
    { window.clearInterval (TemporizadorCambio[i]); }

  // Limpiamos los indicadores
  Rellena (ImagenInterrogacion);
  EscribeReloj ("");

  // Cambiamos el estado de los botones
  BotonNueva.style.display = "inline";
  BotonParar.style.display = "none";
  }

//--------------------------------------------------------------
// Indica éxito o fracaso en el reloj
function RelojEstado (Modo)
  {
  // El color del texto con el que vamos a escribir
  var Color = Modo ? '#0D0' : '#D00';
  DivReloj.style.color = Color;

  // Escribimos el tiempo restante
  var Texto = ("00" + TiempoRestante).slice(-3);
  EscribeReloj (Texto);
  }

//--------------------------------------------------------------
// Escribe un texto en el reloj
function EscribeReloj (Texto)
  { DivReloj.innerHTML = Texto; }

//--------------------------------------------------------------
// Rellena toda la rejilla con una imagen
function Rellena (Imagen)
  {
  ContextoDibujo.fillStyle = "#FFF";
  ContextoDibujo.fillRect (0, 0, Anchura, Altura);
  for ( var Fil=0 ; Fil<MaxFil ; Fil++ )
    {
    for ( var Col=0 ; Col<MaxCol ; Col++ )
      { Pinta (Fil, Col, Imagen); }
    }
  }

//--------------------------------------------------------------
// Borra una celda
function Borra (Fil, Col)
  {
  ContextoDibujo.fillRect (OrigenX+TamanoCelda*Col, OrigenY+TamanoCelda*Fil,
                           TamanoCelda, TamanoCelda);
  }

//--------------------------------------------------------------
// Pinta una celda con una imagen
function Pinta (Fil, Col, Imagen)
  {
  Borra (Fil, Col);
  ContextoDibujo.drawImage (Imagen,
    OrigenX+TamanoCelda*Col, OrigenY+TamanoCelda*Fil);
  }

//--------------------------------------------------------------
// Reacciona a una pulsación del ratón en el dibujo
// http://stackoverflow.com/questions/6430249/getting-mouseclick-events-on-a-canvas
function Pulsacion (Evento)
  {
  // Solo atendemos la pulsación del ratón durante una partida
  if ( EnJuego )
    {
    var Elemento = CanvasDibujo;
    var offsetX = 0, offsetY = 0

    // Si el elemento tiene padre
    if ( Elemento.offsetParent )
      {
      // Sumamos su desplazamiento
      do
        {
        offsetX += Elemento.offsetLeft;
        offsetY += Elemento.offsetTop;
        }
      // Y el de sus ancestros
      while ( Elemento = Elemento.offsetParent );
      }

    // Calculamos las coordenadas del punto de contacto
    var x = Evento.pageX - offsetX;
    var y = Evento.pageY - offsetY;

    // Las convertimos a fila y columna
    var Fil = Math.floor ((y-OrigenY)/TamanoCelda);
    var Col = Math.floor ((x-OrigenX)/TamanoCelda);

    // Procesamos en el juego la pulsación
    if ( Fil>=0 && Fil<MaxFil && Col>=0 && Col<MaxCol &&
         CeldaVacia.indexOf(CeldaNumero(Fil,Col)) == -1 )
      { PulsacionCelda (Fil, Col); }
    }
  }

//--------------------------------------------------------------
// Gestiona la pulsación con el ratón en una celda
function PulsacionCelda (Fil, Col)
  {
  // La celda en la que nos han pulsado
  Celda = CeldaNumero (Fil, Col);

  // Mostramos el símbolo
  Pinta (Fil, Col, ImagenSimbolo[Problema[Celda]]);

  // Si es la primera celda que pulsan de una pareja
  if ( Tanda == 0 )
    {
    // Actualizamos los cambios pendientes
    EjecutaTemporizadores();

    // Guardamos las coordenadas
    Fil1 = Fil;
    Col1 = Col;

    // Anotamos la pulsación
    Tanda = 1 - Tanda;

    // Dentro de cinco segundos anulamos la pulsación
    TemporizadorCambio.push (setTimeout (AnulaPrimeraPulsacion, 5000));
    }

  // Si es la segunda celda que pulsan de una pareja
  else
    {
    // La celda de la primera pulsación
    var Celda1 = CeldaNumero(Fil1,Col1);

    // No admitimos pulsar la segunda vez en la misma celda que la primera
    if ( Celda != Celda1 )
      {
      // Anulamos los temporizadores
      for ( var i=0 ; i<TemporizadorCambio.length ; i++ )
        { window.clearInterval (TemporizadorCambio[i]); }

      // Anotamos la pulsación
      Tanda = 1 - Tanda;

      // Comprobamos si la pareja es correcta
      if ( Problema[Celda1] == Problema[Celda] )
        {
        // Anotamos las celdas que habrá que borrar
        CeldaBien.push (Celda);
        CeldaBien.push (Celda1);

        // Dentro de dos segundos las borramos
        TemporizadorCambio.push (setTimeout (EjecutaTemporizadores, 2000));

        // Anotamos que las celdas ahora están vacías
        CeldaVacia.push (CeldaNumero(Fil1,Col1));
        CeldaVacia.push (CeldaNumero(Fil,Col));

        // Actualizamos el contador de parejas
        ParejasEncontradas++;

        // Quizá hemos terminado la partida
        if ( ParejasEncontradas == TotalProblema )
          { CierraPartida (true); }
        }

      // No han acertado la pareja
      else
        {
        // Penalizamos en tiempo
        Antes -= 1000;

        // Anotamos las celdas que habrá que voltear
        CeldaMal.push (Celda);
        CeldaMal.push (Celda1);

        // Dentro de dos segundos las volteamos
        TemporizadorCambio.push (setTimeout (EjecutaTemporizadores, 2000));
        }
      }
    }
  }

//--------------------------------------------------------------
// Anula la primera pulsación en una celda porque se ha pasado el tiempo
function AnulaPrimeraPulsacion()
  {
  // Cambiamos la tanda
  Tanda = 1 - Tanda;

  // Borramos la imagen
  Pinta (Fil1, Col1, ImagenInterrogacion);
  }

//--------------------------------------------------------------
// Ejecuta la acción de los temporizadores de cambio
function EjecutaTemporizadores()
  {
  var Lugar = [];
  var Fil, Col;

  // Anulamos los temporizadores
  for ( var i=0 ; i<TemporizadorCambio.length ; i++ )
    { window.clearInterval (TemporizadorCambio[i]); }

  // Cambiamos el estado de las celdas
  for ( var i=0 ; i<CeldaBien.length ; i++ )
    {
    Lugar = Posicion(CeldaBien[i]);
    Fil = Lugar[0];
    Col = Lugar[1];
    Borra (Fil, Col);
    }
  for ( var i=0 ; i<CeldaMal.length ; i++ )
    {
    Lugar = Posicion(CeldaMal[i]);
    Fil = Lugar[0];
    Col = Lugar[1];
    Pinta (Fil, Col, ImagenInterrogacion);
    }
  CeldaBien = [];
  CeldaMal = [];
  }

//--------------------------------------------------------------
// Cambiar el valor del reloj
function CambiaReloj()
  {
  // Tomamos nota del tiempo actual
  Ahora = Reloj();

  // Vemos cuánto tiempo ha pasado
  var TiempoEntreLlamadas = Ahora - Antes;

  // Si ha pasado un segundo
  if ( TiempoEntreLlamadas > 1000 )
    {
    // Guardamos el tiempo anterior
    Antes = Ahora;

    // Disminuimos el tiempo disponible
    TiempoRestante--;

    // Puede que se nos haya acabado el tiempo
    if ( TiempoRestante == 0 )
      { CierraPartida (false); }

    // Si no, podemos seguir con la partida
    else
      {
      var Texto = ("00" + TiempoRestante).slice(-3);
      EscribeReloj (Texto);
      }
    }
  }

//--------------------------------------------------------------
// Cierra una partida
function CierraPartida (Exito)
  {
  // Eliminamos el temporizador principal
  window.clearInterval (Temporizador);

  // Eliminamos los temporizadores
  for ( var i=0 ; i<TemporizadorCambio.length ; i++ )
    { window.clearInterval (TemporizadorCambio[i]); }

  // Acabamos la partida
  EnJuego = false;

  // Mostramos iconos de bien o de mal
  if ( Exito )
    { Rellena (ImagenBien); }
  else
    {
    for ( var Fil=0 ; Fil<MaxFil ; Fil++ )
      {
      for ( var Col=0 ; Col<MaxCol ; Col++ )
        {
        if ( CeldaVacia.indexOf(CeldaNumero(Fil, Col)) == -1 )
          { Pinta (Fil, Col, ImagenMal); }
        }
      }
    }

  // El reloj
  RelojEstado (Exito);

  // Cambiamos el estado de los botones
  BotonNueva.style.display = "inline";
  BotonParar.style.display = "none";

  // Guardamos la partida en el historial
  GuardaPartida();
  }

//--------------------------------------------------------------
// Guarda los datos de la partida en el ordenador cliente
function GuardaPartida()
  {
  // Solo podremos guardar la partida si el navegador admite Storage
  if ( typeof(Storage) !== "undefined" )
    {
    // Vemos si ya hemos grabado alguna partida
    if ( localStorage.PSL_Total )
      {
      // Aumentamos el número de partidas almacenadas
      localStorage.PSL_Total = Number(localStorage.PSL_Total) + 1;
      }
    else
      {
      // Ponemos a 1 el número de partidas almacenadas
      localStorage.PSL_Total = 1;
      }

    // Averiguamos la fecha y la hora
    var Momento = new Date();
    var Dia = PonCero (Momento.getDate());
    var Mes = PonCero (Momento.getMonth()+1);
    var Ano = Momento.getFullYear();
    var Hora = PonCero (Momento.getHours());
    var Minuto = PonCero (Momento.getMinutes());
    var Segundo = PonCero (Momento.getSeconds());

    // Añadimos un nuevo registro
    var Clave = 'PSL_' + localStorage.PSL_Total;
    var Valor = Ano + '-' + Mes + '-' + Dia + ',' +
                Hora + ':' + Minuto + ':' + Segundo + ',' + TiempoRestante;
    localStorage.setItem (Clave, Valor);
    }
  }

//--------------------------------------------------------------
// Devuelve un array con la fila y la columna que corresponden
// a una celda
function Posicion (Celda)
  {
  var Respuesta = [];
  Fil = Math.floor (Celda/MaxCol);
  Col = Celda % MaxCol;
  Respuesta[0] = Fil;
  Respuesta[1] = Col;
  return (Respuesta);
  }

//--------------------------------------------------------------
// Dice qué número tiene una celda a partir de sus coordenadas
function CeldaNumero (Fil, Col)
  { return (Fil*MaxCol+Col); }

//--------------------------------------------------------------
// Añade un cero por delante a un número de un solo dígito
function PonCero (Numero)
  {
  if ( Numero < 10 )
    { Numero = '0'+Numero;}
  return Numero;
  }

//--------------------------------------------------------------
// Devuelve un número entero aleatorio entre
// 0 (incluido) y el dado (excluido)
function Aleatorio (Max)
  { return Math.floor(Math.random()*Max); }

//--------------------------------------------------------------
// Revuelve los elementos de un array
// http://bost.ocks.org/mike/shuffle/
function Revuelve (array)
  {
  var m = array.length, t, i;

  // Mientras queden elementos que revolver…
  while (m)
    {
    // Elegimos un elemento de los que quedan…
    i = Math.floor(Math.random() * m--);

    // Y lo intercambiamos con el elemento actual
    t = array[m];
    array[m] = array[i];
    array[i] = t;
    }

  return array;
  }

//--------------------------------------------------------------
// Activa la visión del DIV de Historial
function Historial()
  {
  Cierra();
  DivHistorial.style.display = "block";

  // Preparamos el texto que vamos a mostrar
  var Texto, Clave, Valor, RistraDatos, Fecha, Hora, Puntos;
  if ( localStorage.PSL_Total )
    {
    // Mostramos los registros en una tabla
    Texto = "<TABLE>\n"
    Texto += "<TR>\n"
    Texto += "<TD>Fecha</TD>\n"
    Texto += "<TD>Hora</TD>\n"
    Texto += "<TD>Puntos</TD>\n"
    Texto += "</TR>\n"
    // Cada registro va en una línea
    for ( var i=Number(localStorage.PSL_Total) ; i>0 ; i-- )
      {
      Clave = 'PSL_' + i;
      Valor = localStorage.getItem (Clave);
      RistraDatos = Valor.split (',');
      Fecha = RistraDatos[0];
      Hora = RistraDatos[1];
      Puntos = RistraDatos[2];
      Texto += "<TR>\n";
      Texto += "<TD>" + Fecha + "</TD>\n";
      Texto += "<TD>" + Hora + "</TD>\n";
      Texto += "<TD CLASS=\"centro\">" + Puntos + "</TD>\n";
      Texto += "</TR>\n"
      }
    Texto += "</TABLE>\n"
    }
  else
    { Texto = "No hay ninguna partida en el historial\n"; }

  // Anotamos el texto
  DivListaHistorial.innerHTML = Texto;
  }

//--------------------------------------------------------------
// Borra el historial
function BorraHistorial()
  {
  var Texto;
  Cierra();

  // Si tenemos alguna partida registrada
  if ( localStorage.PSL_Total )
    {
    // Eliminamos todas las claves de los registros
    for ( var i=Number(localStorage.PSL_Total) ; i>0 ; i-- )
      {
      Clave = 'PSL_' + i;
      localStorage.removeItem (Clave);
      }
    // Eliminamos el total de registros
    localStorage.removeItem ("PSL_Total");

    // Aviso al usuario
    Texto = "Se ha borrado el historial";
    }

  else
    { Texto = "No hay nada que borrar"; }

  // Lanzamos el aviso
  alert (Texto);
  }

//--------------------------------------------------------------
// Activa la visión del DIV de Info
function Info()
  {
  Cierra();
  DivInfo.style.display = "block";
  }

//--------------------------------------------------------------
// Activa la visión del DIV de Créditos
function Creditos()
  {
  Cierra();
  DivCreditos.style.display = "block";
  }

//--------------------------------------------------------------
// Activa la visión del DIV de Licencia
function Licencia()
  {
  Cierra();
  DivLicencia.style.display = "block";
  }

//--------------------------------------------------------------
// Activa la visión del DIV de Descarga
function Descarga()
  {
  Cierra();
  DivDescarga.style.display = "block";
  }

//--------------------------------------------------------------
// Activa la visión del DIV de Técnicas
function Tecnicas()
  {
  Cierra();
  DivTecnicas.style.display = "block";
  }

//--------------------------------------------------------------
// Cierra todos los DIV de explicaciones
function Cierra()
  {
  DivInfo.style.display = "none";
  DivHistorial.style.display = "none";
  DivCreditos.style.display = "none";
  DivLicencia.style.display = "none";
  DivDescarga.style.display = "none";
  DivTecnicas.style.display = "none";
  }

//--------------------------------------------------------------
// Devuelve el número de milisegundos desde 1970-01-01
function Reloj()
  { return (new Date()).getTime(); }

//--------------------------------------------------------------
// Precarga una imagen y devuelve su manejador
function PrecargaImagen (Nombre, Tamano)
  {
  // Un nuevo objeto imagen
  var Imagen = new Image();

  // Cuando se cargue lanzamos una notificación
  Imagen.onload = function() { ManejadorCarga (Tamano); };

  // Apuntamos a la URL de la imagen
  Imagen.src = Nombre;

  // Devolvemos el manejador
  return (Imagen);
  }

//--------------------------------------------------------------
// Indica al usuario el transcurso de la carga de imágenes
function ManejadorCarga (Tamano)
  {
  // Añadimos el nuevo tamaño a la cantidad cargada hasta el momento
  CargadoActual += Tamano;

  // Calculamos el porcentaje
  var Porcentaje = Math.floor (100*(CargadoActual/TotalTamano));

  // Lo mostramos en la barra de progreso
  BarraProgreso.value = Porcentaje;

  // Si hemos cargado todas las imágenes
  if ( CargadoActual == TotalTamano )
    // Preparamos la pantalla
    { Prepara(); }
  }

//--------------------------------------------------------------
// Función portable que añade un evento
function AgregaEvento (elemento, evType, funcion, useCapture)
  {
  // Manejador de eventos compatible con navegadores IE5+, NS6 y Mozilla
  // Autor: Scott Andrew
  if ( elemento.addEventListener )
    {
    elemento.addEventListener (evType, funcion, useCapture);
    return true;
    }
  else if ( elemento.attachEvent )
    {
    var r = elemento.attachEvent ('on' + evType, funcion);
    return r;
    }
  else
    { elemento ['on' + evType] = funcion; }
  }

//--------------------------------------------------------------
// La función que inicia todo el sistema
function IniciaTodo()
  {
  // Averiguamos los ID de los elementos
  BotonRueda = document.getElementById ('rueda');
  BotonNueva = document.getElementById ('nueva');
  BotonParar = document.getElementById ('parar');
  BotonInfo = document.getElementById ('info');
  BotonHistorial = document.getElementById ('historial');
  BotonBorrar = document.getElementById ('borrar');
  BotonCreditos = document.getElementById ('creditos');
  BotonLicencia = document.getElementById ('licencia');
  BotonDescarga = document.getElementById ('descarga');
  BotonTecnicas = document.getElementById ('tecnicas');
  BotonCerrar1 = document.getElementById ('botoncerrar1');
  BotonCerrar2 = document.getElementById ('botoncerrar2');
  BotonCerrar3 = document.getElementById ('botoncerrar3');
  BotonCerrar4 = document.getElementById ('botoncerrar4');
  BotonCerrar5 = document.getElementById ('botoncerrar5');
  BotonCerrar6 = document.getElementById ('botoncerrar6');
  CanvasDibujo = document.getElementById ('dibujo');
  ZonaProgreso = document.getElementById ('zonabarra');
  BarraProgreso = document.getElementById ('barra');
  DivReloj = document.getElementById ('divreloj');
  DivInfo = document.getElementById ('divinfo');
  DivHistorial = document.getElementById ('divhistorial');
  DivListaHistorial = document.getElementById ('divlistahistorial');
  DivCreditos = document.getElementById ('divcreditos');
  DivLicencia = document.getElementById ('divlicencia');
  DivDescarga = document.getElementById ('divdescarga');
  DivTecnicas = document.getElementById ('divtecnicas');

  // Vamos presentando algunos controles al usuario
  Inicio();

  // Calculamos cuánto ocupan las imágenes que vamos a precargar
  TotalTamano = TotalSimbolo + 3;

  // Precargamos las imágenes
  CargadoActual = 0;
  ImagenInterrogacion = PrecargaImagen ('imagen/interrogacion.png', 1);
  ImagenBien = PrecargaImagen ('imagen/bien.png', 1);
  ImagenMal = PrecargaImagen ('imagen/mal.png', 1);
  var Texto;
  for ( var i=0 ; i<TotalSimbolo ; i++ )
    {
    Texto = ("0" + i).slice(-2);
    ImagenSimbolo.push (PrecargaImagen ('imagen/simbolo-'+Texto+'.png', 1));
    }

  // Añadimos todos los "listeners" necesarios
  AgregaEvento (BotonNueva, 'click', Nueva, false);
  AgregaEvento (BotonParar, 'click', Parar, false);
  AgregaEvento (BotonInfo, 'click', Info, false);
  AgregaEvento (BotonHistorial, 'click', Historial, false);
  AgregaEvento (BotonBorrar, 'click', BorraHistorial, false);
  AgregaEvento (CanvasDibujo, 'mousedown', Pulsacion, false);
  AgregaEvento (BotonCreditos, 'click', Creditos, false);
  AgregaEvento (BotonLicencia, 'click', Licencia, false);
  AgregaEvento (BotonDescarga, 'click', Descarga, false);
  AgregaEvento (BotonTecnicas, 'click', Tecnicas, false);
  AgregaEvento (BotonCerrar1, 'click', Cierra, false);
  AgregaEvento (BotonCerrar2, 'click', Cierra, false);
  AgregaEvento (BotonCerrar3, 'click', Cierra, false);
  AgregaEvento (BotonCerrar4, 'click', Cierra, false);
  AgregaEvento (BotonCerrar5, 'click', Cierra, false);
  AgregaEvento (BotonCerrar6, 'click', Cierra, false);
  }

//--------------------------------------------------------------
// La función que inicia el sistema cuando se carga la página
AgregaEvento (window, 'load', IniciaTodo, false);
