#!/usr/bin/perl -w

#--------------------------------------------------------------------
# Fichero:  th.pl
# Objetivo: Actualizar las bases de datos y grficos de th
# Autor:    Pedro Reina <pedro@pedroreina.net>
# Fecha:    D.25.9.2011
# Licencia: Dominio pblico
#           http://creativecommons.org/licenses/publicdomain/
#--------------------------------------------------------------------

#---------------------------------
# Declaracin de mdulos
#---------------------------------

use strict;                 # Comprobaciones estrictas
use Device::SerialPort;     # Para manejar el puerto serie

#---------------------------------
# Declaracin de constantes
#---------------------------------

# El directorio donde estn todos los archivos
my $Dir = "/home/web/th/";

# Las variables y sus nombres humamos
my %Nombre =
  (
  "temperatura", "Temperatura",
  "humedad",     "Humedad relativa"
  );

# La orden para ejecutar RRDtool
my $RRDtool = "/usr/bin/rrdtool";

# La orden para actualizar la base de datos SQLite
my $SQLite = "/usr/bin/sqlite3 " . $Dir . "th.db";

# Las dimensiones de los grficos generados
my $Anchura = 600;
my $Altura = 200;

#---------------------------------
# Declaracin de variables
#---------------------------------

my $Temperatura;
my $Humedad;

#---------------------------------
# Programa principal
#---------------------------------

# Para ayudar al desarrollo y depuracin de este programa,
# cada paso individual es una funcin diferente y en cada
# una de ellas se puede activar independientemente la depuracin
# y la ejecucin

# Leemos los valores de las variables
LeerValores();

# Actualizamos las bases de datos RRD
ActualizaBaseDatoRRD();

# Actualizamos la base de datos SQLite
ActualizaBaseDatoSQLite();

# Generamos los grficos diarios
GeneraGraficoDiario();

# Generamos los grficos semanales
GeneraGraficoSemanal();

# Generamos los grficos mensuales
GeneraGraficoMensual();

# Generamos los grficos anuales
GeneraGraficoAnual();

#---------------------------------
# Funciones
#---------------------------------

#--------------------------------------------------------------------
# Funcin:  LeerValores
# Objetivo: Leer los valores de las variables
#--------------------------------------------------------------------

sub LeerValores
  {
  my $Depurar = 0;
  my $Datos;

  # Abrimos el puerto serie
  my $Arduino = Device::SerialPort->new ("/dev/ttyACM0");
  $Arduino->baudrate (9600);

  # Como el Arduino se ha reseteado, esperamos 2 s que vuelva a su ser
  sleep (2);

  # Mandamos un carcter cualquiera al Arduino
  $Arduino->write ('*');

  # Como el Arduino se ha reseteado, esperamos 2 s que vuelva a su ser
  # No s por qu hace falta hacerlo dos veces
  sleep (2);

  # Leemos los datos del Arduino
  $Datos = $Arduino->input();

  # Cerramos el puerto serie
  $Arduino->close();

  # Separamos los dos datos
  $_ = $Datos;
  if ( /(.*) (.*)/ )
    {
    $Temperatura = $1;
    $Humedad = $2;
    }

  if ( $Depurar )
    {
    print "Temperatura: $Temperatura\n";
    print "Humedad: $Humedad\n";
    }
  }

#--------------------------------------------------------------------
# Funcin:  ActualizaBaseDatoRRD
# Objetivo: Actualizar las bases de datos RRD con nuevos valores
#--------------------------------------------------------------------

sub ActualizaBaseDatoRRD
  {
  ActualizaRRD ("temperatura", $Temperatura);
  ActualizaRRD ("humedad", $Humedad);
  }

#--------------------------------------------------------------------
# Funcin:  ActualizaRRD
# Objetivo: Actualizar una base de datos RRD con un valor
#--------------------------------------------------------------------

sub ActualizaRRD
  {
  my $Depurar = 0;
  my $Ejecutar = 1;

  # La base de datos es el primer parmetro
  my $BaseDatos = $_[0];

  # El valor es el segundo parmetro
  my $Valor = $_[1];

  # La orden para hacer la actualizacin
  my $Orden = "$RRDtool update $Dir$BaseDatos.rrd N:$Valor";

  if ( $Depurar )
    { print "$Orden\n"; }

  if ( $Ejecutar )
    { qx ($Orden); }
  }

#--------------------------------------------------------------------
# Funcin:  ActualizaBaseDatoSQLite
# Objetivo: Actualizar la base de datos SQLite
#--------------------------------------------------------------------

sub ActualizaBaseDatoSQLite
  {
  my $Depurar = 0;
  my $Ejecutar = 1;

  my $Consulta;
  my $Orden;

  $Consulta = "\"INSERT INTO th (temperatura, humedad) " .
               "VALUES ($Temperatura, $Humedad)\"";

  $Orden = $SQLite . " " . $Consulta;

  if ( $Depurar )
    { print "$Orden\n"; }

  if ( $Ejecutar )
    { qx ($Orden); }
  }

#--------------------------------------------------------------------
# Funcin:  GeneraGraficoDiario
# Objetivo: Generar los grficos diarios de todas las variables
#--------------------------------------------------------------------

sub GeneraGraficoDiario
  {
  my $Variable;

  while ( $Variable = each (%Nombre) )
    { GeneraGrafico ($Variable, "diario"); }
  }

#--------------------------------------------------------------------
# Funcin:  GeneraGraficoSemanal
# Objetivo: Generar los grficos semanales de todas las variables
#--------------------------------------------------------------------

sub GeneraGraficoSemanal
  {
  my $Variable;

  # Averiguamos la hora local
  my ($seg, $min) = localtime(time);

  # Los datos para obtener los grficos semanales se actualizan
  # cada treinta minutos, as que generamos los grficos a ese ritmo
  if ( $min % 30 == 0 )
    {
    while ( $Variable = each (%Nombre) )
      { GeneraGrafico ($Variable, "semanal"); }
    }
  }

#--------------------------------------------------------------------
# Funcin:  GeneraGraficoMensual
# Objetivo: Generar los grficos mensuales de todas las variables
#--------------------------------------------------------------------

sub GeneraGraficoMensual
  {
  my $Variable;

  # Averiguamos la hora local
  my ($seg, $min, $hora) = localtime(time);

  # Los datos para obtener los grficos mensuales se actualizan
  # cada dos horas, as que generamos los grficos a ese ritmo
  if ( ($min == 0) && ($hora % 2 == 0) )
    {
    while ( $Variable = each (%Nombre) )
      { GeneraGrafico ($Variable, "mensual"); }
    }
  }

#--------------------------------------------------------------------
# Funcin:  GeneraGraficoAnual
# Objetivo: Generar los grficos anuales de todas las variables
#--------------------------------------------------------------------

sub GeneraGraficoAnual
  {
  my $Variable;

  # Averiguamos la hora local
  my ($seg, $min, $hora) = localtime(time);

  # Los datos para obtener los grficos mensuales se actualizan
  # cada da, as que generamos los grficos a ese ritmo
  if ( ($hora == 0) && ($min == 0) )
    {
    while ( $Variable = each (%Nombre) )
      { GeneraGrafico ($Variable, "anual"); }
    }
  }

#--------------------------------------------------------------------
# Funcin:  GeneraGrafico
# Objetivo: Generar un grfico de un tipo de una variable
#--------------------------------------------------------------------

sub GeneraGrafico
  {
  my $Depurar = 0;
  my $Ejecutar = 1;
  my $Orden;
  my $Extension;
  my $EjeHorizontal;

  # La variable es el primer parmetro
  my $Variable = $_[0];

  # El tipo de grfico es el segundo parmetro
  my $Tipo = $_[1];

  # El archivo final del grfico
  my $Archivo = "$Dir$Variable-$Tipo.png";

  # El ttulo del grfico
  my $Titulo = $Nombre{"$Variable"};

  # La extensin en el tiempo y el eje horizontal dependen del tipo
  if ( $Tipo eq "diario" )
    {
    $Extension = "end-28h ";
    $EjeHorizontal = "MINUTE:15:HOUR:1:HOUR:1:0:%H ";
    }
  elsif ( $Tipo eq "semanal" )
    {
    $Extension = "end-9d ";
    $EjeHorizontal = "HOUR:6:DAY:1:DAY:1:86400:%a%t%d ";
    }
  elsif ( $Tipo eq "mensual" )
    {
    $Extension = "end-5w ";
    $EjeHorizontal = "DAY:1:WEEK:1:WEEK:1:604800:Semana%t%V ";
    }
  elsif ( $Tipo eq "anual" )
    {
    $Extension = "end-13mon ";
    $EjeHorizontal = "WEEK:1:MONTH:1:MONTH:1:2592000:%b ";
    }

  # Preparamos la orden para generar el grfico
  $Orden = "$RRDtool graph $Archivo " .
           "--tabwidth 6 " .
           "--title '$Titulo' " .
           "--start $Extension " .
           "--width $Anchura --height $Altura " .
           "--x-grid $EjeHorizontal ";
  $Orden .= "DEF:entrante=$Dir$Variable.rrd:entrante:AVERAGE " .
            "DEF:saliente=$Dir$Variable.rrd:saliente:AVERAGE " .
            "AREA:saliente#5d3fb5:Saliente " .
            "LINE1:entrante#FF0000:Entrante";

  if ( $Depurar )
    { print "$Orden\n"; }

  if ( $Ejecutar )
    { qx ($Orden); }
  }

#--------------------------------------------------------------------
# Funcin:  EjecutaOrden
# Objetivo: Ejecutar una orden del sistema y devolver el resultado
#--------------------------------------------------------------------

sub EjecutaOrden
  {
  # La orden es el primer parmetro
  my $Orden = $_[0];

  # Ejecutamos la orden
  my $Resultado = qx ($Orden);

  # Quitamos el fin de lnea
  chop ($Resultado);

  # Devolvemos el resultado
  return $Resultado;
  }
